<?php
defined( 'ABSPATH' ) || exit;

class HRM_Employees {

    public static function init() {
        add_action( 'wp_ajax_hrm_add_employee', [ __CLASS__, 'handle_ajax_add_employee' ] );
        add_action( 'wp_ajax_hrm_edit_employee', [ __CLASS__, 'handle_ajax_edit_employee' ] );
        add_action( 'wp_ajax_hrm_delete_employee', [ __CLASS__, 'handle_ajax_delete_employee' ] );
        add_action( 'wp_ajax_hrm_toggle_status', [ __CLASS__, 'handle_ajax_toggle_status' ] );
        add_action( 'wp_ajax_hrm_reset_password', [ __CLASS__, 'handle_ajax_reset_password' ] );
        add_action( 'wp_ajax_hrm_sync_employees', [ __CLASS__, 'handle_ajax_sync_employees' ] ); // New Sync Handler
    }

    public static function render_employees() {
        if ( isset( $_GET['action'] ) && $_GET['action'] === 'add_new' ) {
            self::render_add_employee_page();
            return;
        }

        $search = isset( $_GET['s'] ) ? sanitize_text_field( $_GET['s'] ) : '';
        $role_filter = isset( $_GET['role'] ) ? sanitize_text_field( $_GET['role'] ) : '';
        $dept_filter = isset( $_GET['dept'] ) ? sanitize_text_field( $_GET['dept'] ) : '';

        // Pagination
        $per_page = HRM_Pagination::get_per_page();
        $current_page = HRM_Pagination::get_current_page();

        // Query Employees
        $args = [
            'post_type' => HRM_Setup::EMPLOYEE_POST_TYPE,
            'posts_per_page' => $per_page,
            'paged' => $current_page,
            's' => $search,
            'post_status' => ['publish', 'draft']
        ];

        // Department filter
        if ( $dept_filter ) {
            $args['meta_query'] = [
                [
                    'key' => 'hrm_department',
                    'value' => $dept_filter,
                    'compare' => '='
                ]
            ];
        }
        
        // Stats Logic
        $employee_count_obj = wp_count_posts( HRM_Setup::EMPLOYEE_POST_TYPE );
        $total_employees = 0;
        if (isset($employee_count_obj->publish)) {
            $total_employees += (int)$employee_count_obj->publish;
        }
        if (isset($employee_count_obj->draft)) {
            $total_employees += (int)$employee_count_obj->draft;
        }
        $active_count = isset($employee_count_obj->publish) ? (int)$employee_count_obj->publish : 0;
        
        // Get unique departments
        global $wpdb;
        $departments = $wpdb->get_col( "SELECT DISTINCT meta_value FROM {$wpdb->postmeta} WHERE meta_key = 'hrm_department' AND meta_value != ''" );
        $dept_count = count( array_unique($departments) );

        $query = new WP_Query( $args );
        $total_employees_query = $query->found_posts;
        ?>
        <div class="wrap hrm-wrap hrm-modern-dashboard">
            <!-- Header -->
            <div class="hrm-header-section">
                <div class="hrm-header-left">
                     <h1>Employees</h1>
                     <p>Manage your workforce and team structure.</p>
                </div>
                <div class="hrm-header-right">
                    <button class="hrm-btn-outline" id="hrm-sync-employees" style="margin-right: 10px;">
                        <span class="dashicons dashicons-update"></span> Sync Users
                    </button>
                    <a href="<?php echo admin_url('admin.php?page=hrm-employees&action=add_new'); ?>" class="hrm-btn-primary">+ Add Employee</a>
                </div>
            </div>

            <!-- Stats Row -->
            <div class="hrm-stats-row">
                <div class="hrm-stat-card white">
                    <div class="stat-header">
                        <span>Total Employees</span>
                        <span class="dashicons dashicons-groups" style="color: var(--primary-color);"></span>
                    </div>
                    <div class="stat-number"><?php echo $total_employees; ?></div>
                    <div class="stat-footer">
                        <span class="badge-green">Active</span>
                    </div>
                </div>
                <div class="hrm-stat-card white">
                    <div class="stat-header">
                        <span>Departments</span>
                        <span class="dashicons dashicons-networking" style="color: var(--info-color);"></span>
                    </div>
                    <div class="stat-number"><?php echo $dept_count; ?></div>
                    <div class="stat-footer">
                        <span class="text-muted">Organizational Units</span>
                    </div>
                </div>
                <!-- Placeholder for future stats -->
                <div class="hrm-stat-card white">
                     <div class="stat-header">
                        <span>New This Month</span>
                        <span class="dashicons dashicons-calendar-alt" style="color: var(--warning-color);"></span>
                    </div>
                    <div class="stat-number">
                        <?php 
                        // Approximate count based on publish date
                        $new_joiners = new WP_Query([
                            'post_type' => HRM_Setup::EMPLOYEE_POST_TYPE,
                            'date_query' => ['after' => '1 month ago']
                        ]);
                        echo $new_joiners->found_posts; 
                        ?>
                    </div>
                    <div class="stat-footer">
                        <span class="badge-blue">Growing</span>
                    </div>
                </div>
            </div>

            <!-- Filter Bar -->
            <div class="hrm-filter-bar hrm-card" style="margin-bottom: 24px;">
                <input type="text" id="hrm-search-employee" class="hrm-search-input" placeholder="Search by name, role..." value="<?php echo esc_attr($search); ?>">
                
                <select id="hrm-dept-filter" class="hrm-select-input">
                    <option value="">All Departments</option>
                    <?php foreach($departments as $dept): ?>
                        <option value="<?php echo esc_attr($dept); ?>" <?php selected( isset($_GET['dept']) && $_GET['dept'] == $dept ); ?>><?php echo esc_html($dept); ?></option>
                    <?php endforeach; ?>
                </select>

                <button class="hrm-btn-outline" onclick="window.location.reload()"><span class="dashicons dashicons-filter"></span> Reset</button>
            </div>

            <!-- Employee Grid -->
            <div class="hrm-employee-grid">
                <?php if ( $query->have_posts() ) : while ( $query->have_posts() ) : $query->the_post(); 
                    $id = get_the_ID();
                    $position = get_post_meta( $id, 'hrm_position', true ) ?: 'Employee';
                    $department = get_post_meta( $id, 'hrm_department', true ) ?: 'General';
                    $email = get_post_meta( $id, 'hrm_email', true );
                    $status_class = get_post_status() === 'publish' ? 'active' : 'inactive';
                    
                    $profile_pic_id = get_post_meta( $id, 'hrm_profile_pic', true );
                ?>
                <div class="hrm-employee-card hrm-card" data-name="<?php echo strtolower(get_the_title()); ?>" data-department="<?php echo esc_attr($department); ?>">
                    <div class="emp-cover-bg"></div>
                    
                    <div class="emp-card-header-actions">
                         <span class="hrm-status-ind <?php echo $status_class; ?>" title="<?php echo ucfirst($status_class); ?>"></span>
                    </div>

                    <div class="emp-avatar-lg">
                        <?php 
                        if ( $profile_pic_id ) {
                            echo wp_get_attachment_image( $profile_pic_id, 'thumbnail', false, ['class' => 'emp-avatar-img'] );
                        } else {
                            echo '<div class="emp-avatar-placeholder">' . strtoupper(substr(get_the_title(), 0, 1)) . '</div>'; 
                        }
                        ?>
                    </div>
                    
                    <div class="emp-body">
                        <h3 class="emp-name"><?php the_title(); ?></h3>
                        <div class="emp-role"><?php echo esc_html($position); ?></div>
                        <div class="emp-dept-badge"><?php echo esc_html($department); ?></div>
                        
                        <div class="emp-meta-info">
                            <?php if($email): ?>
                                <span title="<?php echo esc_attr($email); ?>"><i class="dashicons dashicons-email"></i> <?php echo esc_html($email); ?></span>
                            <?php endif; ?>
                        </div>
                    </div>

                    <div class="emp-footer">
                        <div class="emp-main-actions">
                             <a href="<?php echo admin_url('admin.php?page=hrm-employee-profile&id='.$id); ?>" class="hrm-btn-primary btn-sm btn-block">View Profile</a>
                        </div>
                        <div class="emp-secondary-actions">
                             <button class="hrm-btn-icon hrm-edit-employee" 
                                data-id="<?php echo $id; ?>" 
                                data-name="<?php echo esc_attr(get_the_title()); ?>"
                                data-email="<?php echo esc_attr($email); ?>"
                                data-position="<?php echo esc_attr($position); ?>"
                                data-department="<?php echo esc_attr($department); ?>"
                                data-phone="<?php echo get_post_meta($id, 'hrm_phone', true); ?>"
                                data-username="<?php $u = get_user_by('id', get_post_field('post_author', $id)); echo $u ? esc_attr($u->user_login) : ''; ?>"
                                data-joining-date="<?php echo esc_attr(get_post_meta($id, 'hrm_joining_date', true)); ?>"
                                data-profile-pic="<?php echo esc_attr($profile_pic_id); ?>"
                                data-profile-pic-url="<?php echo $profile_pic_id ? esc_url(wp_get_attachment_image_url($profile_pic_id, 'thumbnail')) : ''; ?>"
                                data-leaves-allowed="<?php echo esc_attr(get_post_meta($id, 'hrm_leaves_allowed', true)); ?>"
                                data-leave-applicability="<?php echo esc_attr(get_post_meta($id, 'hrm_leave_applicability', true)); ?>"
                                data-probation-period="<?php echo esc_attr(get_post_meta($id, 'hrm_probation_period', true)); ?>"
                                title="Edit Details">
                                <span class="dashicons dashicons-edit"></span>
                             </button>

                             <button class="hrm-btn-icon hrm-reset-password-trigger" data-id="<?php echo $id; ?>" data-name="<?php echo esc_attr(get_the_title()); ?>" title="Reset Password">
                                <span class="dashicons dashicons-lock"></span>
                             </button>

                            <?php if($status_class === 'active'): ?>
                                <button class="hrm-btn-icon hrm-toggle-status warning" data-id="<?php echo $id; ?>" data-status="draft" title="Deactivate Account">
                                    <span class="dashicons dashicons-hidden"></span>
                                </button>
                            <?php else: ?>
                                <button class="hrm-btn-icon hrm-toggle-status success" data-id="<?php echo $id; ?>" data-status="publish" title="Activate Account">
                                    <span class="dashicons dashicons-visibility"></span>
                                </button>
                            <?php endif; ?>
                            
                            <?php if ( current_user_can('manage_options') ): ?>
                            <button class="hrm-btn-icon hrm-delete-employee danger" data-id="<?php echo $id; ?>" title="Delete Employee">
                                <span class="dashicons dashicons-trash"></span>
                            </button>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
                <?php endwhile; wp_reset_postdata(); else : ?>
                <div class="hrm-no-data" style="grid-column: 1/-1;">No employees found.</div>
                <?php endif; ?>
            </div>
            
            <!-- Pagination -->
            <?php 
            HRM_Pagination::render([
                'total_items' => $total_employees_query,
                'current_page' => $current_page,
                'per_page' => $per_page,
                'query_args' => [
                    's' => $search,
                    'dept' => $dept_filter,
                    'role' => $role_filter
                ],
                'show_sizes' => true,
                'context' => 'admin'
            ]);
            HRM_Pagination::render_script();
            ?>
        </div>

        <!-- Render Modals -->
        <?php self::render_edit_modal(); ?>
        <?php self::render_reset_password_modal(); ?>
        <script>
        jQuery(document).ready(function($) {
            // Edit Modal Trigger
            $('.hrm-edit-employee').on('click', function() {
                var btn = $(this);
                $('#edit-employee-id').val( btn.data('id') );
                $('#edit-name').val( btn.data('name') );
                $('#edit-email').val( btn.data('email') );
                $('#edit-position').val( btn.data('position') );
                $('#edit-department').val( btn.data('department') );
                $('#edit-phone').val( btn.data('phone') );
                $('#edit-username').val( btn.data('username') );
                $('#edit-joining-date').val( btn.data('joining-date') || '' );
                $('#edit-current-profile-pic').val( btn.data('profile-pic') || '' );
                
                // Leave & Probation fields
                $('#edit-leaves-allowed').val( btn.data('leaves-allowed') || 0 );
                $('#edit-probation-period').val( btn.data('probation-period') || 0 );
                $('#edit-leave-applicability').val( btn.data('leave-applicability') || 'after_probation' );
                
                // Reset profile picture preview
                var profilePicUrl = btn.data('profile-pic-url');
                var previewContainer = $('#edit-profile-preview');
                if (profilePicUrl) {
                    previewContainer.html('<img src="' + profilePicUrl + '" alt="Profile">');
                    previewContainer.addClass('has-image');
                    $('#edit-remove-photo-btn').show();
                } else {
                    previewContainer.html('<span class="dashicons dashicons-admin-users"></span>');
                    previewContainer.removeClass('has-image');
                    $('#edit-remove-photo-btn').hide();
                }
                
                // Reset file input
                $('#edit-profile-picture-input').val('');
                
                $('#hrm-edit-modal').fadeIn();
            });

            $('.hrm-close-modal').on('click', function() {
                $('.hrm-modal').fadeOut();
            });
            
            // Edit Modal - Profile Picture Upload
            $('#edit-upload-photo-btn').on('click', function() {
                $('#edit-profile-picture-input').click();
            });
            
            $('#edit-profile-picture-input').on('change', function() {
                var file = this.files[0];
                if (file) {
                    // Validate file type
                    var allowedTypes = ['image/jpeg', 'image/png', 'image/gif', 'image/webp'];
                    if (!allowedTypes.includes(file.type)) {
                        alert('Please upload a valid image file (JPG, PNG, GIF, or WebP)');
                        this.value = '';
                        return;
                    }
                    
                    // Validate file size (max 2MB)
                    if (file.size > 2 * 1024 * 1024) {
                        alert('File size must be less than 2MB');
                        this.value = '';
                        return;
                    }
                    
                    // Preview image
                    var reader = new FileReader();
                    reader.onload = function(e) {
                        $('#edit-profile-preview').html('<img src="' + e.target.result + '" alt="Profile Preview">');
                        $('#edit-profile-preview').addClass('has-image');
                        $('#edit-remove-photo-btn').show();
                    }
                    reader.readAsDataURL(file);
                }
            });
            
            $('#edit-remove-photo-btn').on('click', function() {
                $('#edit-profile-picture-input').val('');
                $('#edit-current-profile-pic').val('remove');
                $('#edit-profile-preview').html('<span class="dashicons dashicons-admin-users"></span>');
                $('#edit-profile-preview').removeClass('has-image');
                $(this).hide();
            });
            
            // Delete Action
            $('.hrm-delete-employee').on('click', function() {
                if(!confirm('Are you sure you want to delete this user? This will remove all account data, tasks, and attendance records.')) return;
                
                var btn = $(this);
                btn.prop('disabled', true).text('Deleting...');
                
                $.post(ajaxurl, {
                    action: 'hrm_delete_employee',
                    employee_id: btn.data('id')
                }, function(res) {
                    if(res.success) {
                        hrmToast('success', 'User deleted successfully.');
                        setTimeout(function(){ location.reload(); }, 1000);
                    } else {
                        alert(res.data);
                        btn.prop('disabled', false).text('Delete Account');
                    }
                });
            });

            // Sync Action
            $('#hrm-sync-employees').on('click', function() {
                var btn = $(this);
                btn.prop('disabled', true).html('<span class="dashicons dashicons-update"></span> Syncing...');
                
                $.post(ajaxurl, {
                    action: 'hrm_sync_employees',
                    nonce: '<?php echo wp_create_nonce('hrm_ajax_nonce'); ?>'
                }, function(res) {
                    if(res.success) {
                        hrmToast('success', res.data);
                        setTimeout(function(){ location.reload(); }, 1000);
                    } else {
                        hrmToast('error', res.data);
                        btn.prop('disabled', false).html('<span class="dashicons dashicons-update"></span> Sync Users');
                    }
                });
            });

            // Handle Edit Form Submit with FormData
            $('#hrm-edit-employee-form').on('submit', function(e) {
                e.preventDefault();
                var btn = $('#edit-submit-btn');
                var originalHtml = btn.html();
                btn.html('<span class="dashicons dashicons-update spin"></span> Saving...').prop('disabled', true);

                var formData = new FormData(this);
                
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: formData,
                    processData: false,
                    contentType: false,
                    success: function(res) {
                        if(res.success) {
                            hrmToast('success', res.data);
                            setTimeout(function(){ location.reload(); }, 1000);
                        } else {
                            hrmToast('error', res.data);
                            btn.html(originalHtml).prop('disabled', false);
                        }
                    },
                    error: function() {
                        hrmToast('error', 'An error occurred. Please try again.');
                        btn.html(originalHtml).prop('disabled', false);
                    }
                });
            });
            
            // Handle Add Form Submit
            $('#hrm-add-employee-form').on('submit', function(e) {
                e.preventDefault();
                var btn = $(this).find('button[type="submit"]');
                var originalText = btn.text();
                btn.text('Creating...').prop('disabled', true);

                $.post(ajaxurl, $(this).serialize(), function(res) {
                     if(res.success) {
                        hrmToast('success', res.data);
                        setTimeout(function(){ window.location.href = '<?php echo admin_url('admin.php?page=hrm-employees'); ?>'; }, 1000);
                    } else {
                        hrmToast('error', res.data);
                        btn.text(originalText).prop('disabled', false);
                    }
                });
            });

            // Toggle Status
            $('.hrm-toggle-status').on('click', function() {
                var btn = $(this);
                // btn.text('Updating...');
                $.post(ajaxurl, {
                    action: 'hrm_toggle_status',
                    employee_id: btn.data('id'),
                    status: btn.data('status')
                }, function(res) {
                    if(res.success) {
                        hrmToast('success', 'Status updated successfully.');
                        location.reload();
                    }
                });
            });

            // Reset Password Trigger
            $('.hrm-reset-password-trigger').on('click', function() {
                var btn = $(this);
                $('#reset-pwd-id').val( btn.data('id') );
                $('#reset-pwd-name').text( btn.data('name') );
                $('#hrm-reset-password-modal').fadeIn();
            });

            // Handle Reset Password Submit
            $('#hrm-admin-reset-pwd-form').on('submit', function(e) {
                e.preventDefault();
                var btn = $(this).find('button[type="submit"]');
                var originalText = btn.text();
                btn.text('Updating...').prop('disabled', true);

                $.post(ajaxurl, $(this).serialize(), function(res) {
                     if(res.success) {
                        hrmToast('success', 'Password updated successfully.');
                        $('#hrm-reset-password-modal').fadeOut();
                        $('#hrm-admin-reset-pwd-form')[0].reset();
                        btn.text(originalText).prop('disabled', false);
                    } else {
                        hrmToast('error', res.data);
                        btn.text(originalText).prop('disabled', false);
                    }
                });
            });

            // Global Toast Function
            window.hrmToast = function(type, message) {
                var icon = type === 'success' ? 'dashicons-yes-alt' : 'dashicons-warning';
                var html = `
                <div class="hrm-toast-notification ${type}">
                    <span class="dashicons ${icon}"></span>
                    <span>${message}</span>
                    <button class="hrm-toast-close" onclick="this.parentElement.remove()">&times;</button>
                </div>`;
                
                $('body').append(html);
                setTimeout(function() {
                    $('.hrm-toast-notification').fadeOut(function() { $(this).remove(); });
                }, 4000);
            };

            window.openModal = function(id) {
                $('#' + id).css('display', 'flex').hide().fadeIn();
            };

            window.closeModal = function(id) {
                $('#' + id).fadeOut();
            };

        });
        </script>
        <?php
    }

    private static function render_edit_modal() {
        ?>
        <div id="hrm-edit-modal" class="hrm-modal" style="display:none;">
            <div class="hrm-modal-content hrm-premium-modal hrm-edit-modal-lg">
                <div class="hrm-modal-header-premium">
                    <div class="header-icon-wrapper">
                        <span class="dashicons dashicons-edit"></span>
                    </div>
                    <div class="header-text">
                        <h2>Edit Employee</h2>
                        <p>Update employee details and profile information</p>
                    </div>
                     <button class="hrm-modal-close-premium" onclick="closeModal('hrm-edit-modal')">
                         <span class="dashicons dashicons-no-alt"></span>
                     </button>
                </div>
                
                <form id="hrm-edit-employee-form" class="hrm-premium-form-modal" enctype="multipart/form-data">
                    <input type="hidden" name="action" value="hrm_edit_employee">
                    <input type="hidden" name="employee_id" id="edit-employee-id">
                    <input type="hidden" name="current_profile_pic" id="edit-current-profile-pic">
                    <?php wp_nonce_field( 'hrm_ajax_nonce', 'nonce' ); ?>
                    
                    <div class="hrm-edit-form-layout">
                        <!-- Left Side - Profile Picture -->
                        <div class="hrm-edit-form-sidebar">
                            <div class="edit-profile-upload">
                                <div class="edit-profile-preview" id="edit-profile-preview">
                                    <span class="dashicons dashicons-admin-users"></span>
                                </div>
                                <input type="file" name="profile_picture" id="edit-profile-picture-input" accept="image/jpeg,image/png,image/gif,image/webp" style="display:none;">
                                <div class="edit-upload-actions">
                                    <button type="button" class="hrm-btn-outline btn-sm" id="edit-upload-photo-btn">
                                        <span class="dashicons dashicons-camera"></span><span class="btn-text">Change Photo</span>
                                    </button>
                                    <button type="button" class="hrm-btn-ghost btn-sm danger" id="edit-remove-photo-btn" style="display:none;">
                                        <span class="dashicons dashicons-trash"></span><span class="btn-text">Remove</span>
                                    </button>
                                </div>
                                <p class="upload-hint">Max 2MB. JPG, PNG, GIF, WebP</p>
                            </div>
                            
                            <!-- Joining Date -->
                            <div class="edit-joining-section">
                                <label><span class="dashicons dashicons-calendar-alt"></span> Joining Date</label>
                                <input type="date" name="joining_date" id="edit-joining-date">
                            </div>
                        </div>
                        
                        <!-- Right Side - Form Fields -->
                        <div class="hrm-edit-form-main">
                            <!-- Personal Info -->
                            <div class="hrm-form-section-compact">
                                <h5 class="section-label"><span class="dashicons dashicons-id"></span> Personal Information</h5>
                                <div class="hrm-form-group full-width">
                                    <label>Full Name</label>
                                    <input type="text" name="name" id="edit-name" required>
                                </div>
                                
                                <div class="hrm-row-modal">
                                     <div class="hrm-form-group">
                                         <label>Email</label>
                                         <input type="email" name="email" id="edit-email" required>
                                    </div>
                                    <div class="hrm-form-group">
                                         <label>Username</label>
                                         <input type="text" id="edit-username" readonly disabled class="input-readonly">
                                    </div>
                                </div>
                            </div>

                            <!-- Role & Department -->
                            <div class="hrm-form-section-compact">
                                <h5 class="section-label"><span class="dashicons dashicons-businessperson"></span> Role & Contact</h5>
                                 <div class="hrm-row-modal">
                                    <div class="hrm-form-group">
                                        <label>Position</label>
                                        <input type="text" name="position" id="edit-position">
                                    </div>
                                    <div class="hrm-form-group">
                                        <label>Department</label>
                                        <input type="text" name="department" id="edit-department">
                                    </div>
                                </div>
                                
                                 <div class="hrm-form-group full-width">
                                     <label>Phone Number</label>
                                     <input type="tel" name="phone" id="edit-phone">
                                </div>
                            </div>

                            <!-- Leave & Probation Section -->
                            <div class="hrm-form-section-compact">
                                <h5 class="section-label"><span class="dashicons dashicons-calendar-alt"></span> Leave & Probation Policy</h5>
                                <div class="hrm-row-modal">
                                    <div class="hrm-form-group">
                                        <label>Leaves Allowed</label>
                                        <input type="number" name="leaves_allowed" id="edit-leaves-allowed" min="0" max="365" step="1" placeholder="e.g. 20">
                                    </div>
                                    <div class="hrm-form-group">
                                        <label>Probation Period (Months)</label>
                                        <input type="number" name="probation_period" id="edit-probation-period" min="0" max="24" step="1" placeholder="e.g. 3">
                                    </div>
                                </div>
                                <div class="hrm-form-group full-width">
                                    <label>Leave Applicability</label>
                                    <select name="leave_applicability" id="edit-leave-applicability" class="hrm-select-input">
                                        <option value="after_probation">After Probation</option>
                                        <option value="during_probation">During Probation</option>
                                    </select>
                                </div>
                            </div>
                        </div>
                    </div>

                    <div class="hrm-modal-footer-premium">
                        <button type="button" class="hrm-btn-ghost" onclick="closeModal('hrm-edit-modal')">Cancel</button>
                        <button type="submit" class="hrm-btn-primary" id="edit-submit-btn">
                            <span class="dashicons dashicons-saved"></span> Save Changes
                        </button>
                    </div>
                </form>
            </div>
        </div>
        <?php
    }

    private static function render_reset_password_modal() {
        ?>
        <div id="hrm-reset-password-modal" class="hrm-modal" style="display:none;">
            <div class="hrm-modal-content hrm-premium-modal">
                 <div class="hrm-modal-header-premium">
                    <div class="header-icon-wrapper">
                        <span class="dashicons dashicons-lock"></span>
                    </div>
                    <div class="header-text">
                        <h2>Reset Password</h2>
                        <p>Set a new secure password for <strong id="reset-pwd-name">User</strong>.</p>
                    </div>
                     <button class="hrm-modal-close-premium" onclick="closeModal('hrm-reset-password-modal')">
                         <span class="dashicons dashicons-no-alt"></span>
                     </button>
                </div>

                <form id="hrm-admin-reset-pwd-form" class="hrm-premium-form-modal">
                    <input type="hidden" name="action" value="hrm_reset_password">
                    <input type="hidden" name="employee_id" id="reset-pwd-id">
                    
                    <div class="hrm-form-section">
                        <div class="hrm-form-group full-width">
                            <label>New Password <span class="required">*</span></label>
                            <input type="text" name="password" required minlength="6" placeholder="Enter new password">
                        </div>
                    </div>

                    <div class="hrm-modal-footer-premium">
                        <button type="button" class="hrm-btn-ghost" onclick="closeModal('hrm-reset-password-modal')">Cancel</button>
                        <button type="submit" class="hrm-btn-primary">Update Password</button>
                    </div>
                </form>
            </div>
        </div>
        <?php
    }

    public static function render_add_employee_page() {
        ?>
        <div class="wrap hrm-wrap hrm-modern-dashboard hrm-add-employee-wrap">
            <!-- Back Navigation -->
            <div class="hrm-breadcrumb-nav">
                <a href="<?php echo admin_url('admin.php?page=hrm-employees'); ?>" class="hrm-back-link">
                    <span class="dashicons dashicons-arrow-left-alt2"></span>
                    <span>Back to Employees</span>
                </a>
            </div>
            
            <!-- Main Form Container -->
            <div class="hrm-add-employee-container">
                <!-- Premium Header -->
                <div class="hrm-add-employee-header">
                    <div class="header-visual">
                        <div class="header-icon-box">
                            <span class="dashicons dashicons-admin-users"></span>
                        </div>
                        <div class="header-pattern"></div>
                    </div>
                    <div class="header-content">
                        <h1>Add New Employee</h1>
                        <p>Create a new team member account with personal info, role details, and payroll configuration</p>
                    </div>
                    <div class="header-progress">
                        <div class="progress-step active" data-step="1">
                            <span class="step-dot">1</span>
                            <span class="step-label">Personal</span>
                        </div>
                        <div class="progress-line"></div>
                        <div class="progress-step" data-step="2">
                            <span class="step-dot">2</span>
                            <span class="step-label">Role</span>
                        </div>
                        <div class="progress-line"></div>
                        <div class="progress-step" data-step="3">
                            <span class="step-dot">3</span>
                            <span class="step-label">Payroll</span>
                        </div>
                        <div class="progress-line"></div>
                        <div class="progress-step" data-step="4">
                            <span class="step-dot">4</span>
                            <span class="step-label">Security</span>
                        </div>
                    </div>
                </div>
                
                <form id="hrm-add-employee-form" class="hrm-premium-employee-form" enctype="multipart/form-data">
                    <input type="hidden" name="action" value="hrm_add_employee">
                    <?php wp_nonce_field( 'hrm_ajax_nonce', 'nonce' ); ?>
                    
                    <div class="hrm-form-layout-modern">
                        <!-- Left Sidebar - Profile & Quick Info -->
                        <aside class="hrm-form-sidebar-modern">
                            <!-- Profile Picture Upload -->
                            <div class="hrm-sidebar-card">
                                <div class="sidebar-card-header">
                                    <span class="dashicons dashicons-camera"></span>
                                    <h4>Profile Picture</h4>
                                </div>
                                <div class="profile-upload-modern">
                                    <div class="profile-preview-modern" id="profile-preview">
                                        <span class="dashicons dashicons-admin-users"></span>
                                        <div class="preview-overlay">
                                            <span class="dashicons dashicons-camera-alt"></span>
                                        </div>
                                    </div>
                                    <input type="file" name="profile_picture" id="profile-picture-input" accept="image/jpeg,image/png,image/gif,image/webp" style="display:none;">
                                    <div class="upload-actions-modern">
                                        <button type="button" class="hrm-btn-upload" id="upload-photo-btn">
                                            <span class="dashicons dashicons-upload"></span>
                                            <span class="btn-text">Upload Photo</span>
                                        </button>
                                        <button type="button" class="hrm-btn-remove" id="remove-photo-btn" style="display:none;">
                                            <span class="dashicons dashicons-trash"></span>
                                        </button>
                                    </div>
                                    <p class="upload-hint-modern">JPG, PNG, GIF, or WebP. Max 2MB.</p>
                                </div>
                            </div>
                            
                            <!-- Employment Details -->
                            <div class="hrm-sidebar-card">
                                <div class="sidebar-card-header">
                                    <span class="dashicons dashicons-calendar-alt"></span>
                                    <h4>Employment</h4>
                                </div>
                                <div class="hrm-form-group-modern">
                                    <label>Joining Date <span class="required">*</span></label>
                                    <div class="input-with-icon">
                                        <span class="input-icon dashicons dashicons-calendar-alt"></span>
                                        <input type="date" name="joining_date" id="joining-date" required value="<?php echo date('Y-m-d'); ?>">
                                    </div>
                                    <span class="field-hint">Official start date</span>
                                </div>
                            </div>

                            <!-- Quick Stats Preview -->
                            <div class="hrm-sidebar-card hrm-preview-card">
                                <div class="sidebar-card-header">
                                    <span class="dashicons dashicons-visibility"></span>
                                    <h4>Preview</h4>
                                </div>
                                <div class="employee-preview-mini">
                                    <div class="preview-avatar" id="preview-avatar-mini">
                                        <span class="dashicons dashicons-admin-users"></span>
                                    </div>
                                    <div class="preview-info">
                                        <span class="preview-name" id="preview-name">Employee Name</span>
                                        <span class="preview-role" id="preview-role">Position • Department</span>
                                    </div>
                                </div>
                            </div>
                        </aside>
                        
                        <!-- Main Form Content -->
                        <main class="hrm-form-main-modern">
                            <!-- Section 1: Personal Details -->
                            <section class="hrm-form-section-modern" data-section="1">
                                <div class="section-header-modern">
                                    <div class="section-icon-box personal">
                                        <span class="dashicons dashicons-id"></span>
                                    </div>
                                    <div class="section-title-group">
                                        <h3>Personal Information</h3>
                                        <p>Basic identity and contact information</p>
                                    </div>
                                    <span class="section-badge">Required</span>
                                </div>
                                
                                <div class="form-grid-modern">
                                    <div class="form-group-modern">
                                        <label class="form-label-modern">
                                            Full Name <span class="required">*</span>
                                        </label>
                                        <div class="input-wrapper-modern">
                                            <span class="input-prefix-icon dashicons dashicons-admin-users"></span>
                                            <input type="text" name="name" id="input-name" required placeholder="Enter full name" class="input-modern">
                                        </div>
                                    </div>
                                    
                                    <div class="form-group-modern">
                                        <label class="form-label-modern">
                                            Username <span class="required">*</span>
                                        </label>
                                        <div class="input-wrapper-modern">
                                            <span class="input-prefix-icon dashicons dashicons-admin-users"></span>
                                            <input type="text" name="username" required placeholder="Choose username" class="input-modern">
                                            <span class="input-suffix">@company</span>
                                        </div>
                                        <span class="field-hint">Used for login • Cannot be changed later</span>
                                    </div>
                                    
                                    <div class="form-group-modern full-width">
                                        <label class="form-label-modern">
                                            Email Address <span class="required">*</span>
                                        </label>
                                        <div class="input-wrapper-modern">
                                            <span class="input-prefix-icon dashicons dashicons-email"></span>
                                            <input type="email" name="email" required placeholder="employee@company.com" class="input-modern">
                                        </div>
                                        <span class="field-hint">Primary contact and login credential</span>
                                    </div>
                                </div>
                            </section>

                            <!-- Section 2: Role & Contact -->
                            <section class="hrm-form-section-modern" data-section="2">
                                <div class="section-header-modern">
                                    <div class="section-icon-box role">
                                        <span class="dashicons dashicons-businessperson"></span>
                                    </div>
                                    <div class="section-title-group">
                                        <h3>Role & Department</h3>
                                        <p>Job position and organizational details</p>
                                    </div>
                                </div>
                                
                                <div class="form-grid-modern">
                                    <div class="form-group-modern">
                                        <label class="form-label-modern">Position / Job Title</label>
                                        <div class="input-wrapper-modern">
                                            <span class="input-prefix-icon dashicons dashicons-portfolio"></span>
                                            <input type="text" name="position" id="input-position" placeholder="e.g. Software Engineer" class="input-modern">
                                        </div>
                                    </div>
                                    
                                    <div class="form-group-modern">
                                        <label class="form-label-modern">Department</label>
                                        <div class="input-wrapper-modern">
                                            <span class="input-prefix-icon dashicons dashicons-networking"></span>
                                            <input type="text" name="department" id="input-department" placeholder="e.g. Engineering" class="input-modern">
                                        </div>
                                    </div>
                                    
                                    <div class="form-group-modern full-width">
                                        <label class="form-label-modern">Phone Number</label>
                                        <div class="input-wrapper-modern">
                                            <span class="input-prefix-icon dashicons dashicons-phone"></span>
                                            <input type="tel" name="phone" placeholder="+1 234 567 8900" class="input-modern">
                                        </div>
                                    </div>
                                </div>
                            </section>

                            <!-- Section 3: Payroll & Compensation (NEW) -->
                            <section class="hrm-form-section-modern hrm-payroll-section" data-section="3">
                                <div class="section-header-modern">
                                    <div class="section-icon-box payroll">
                                        <span class="dashicons dashicons-money-alt"></span>
                                    </div>
                                    <div class="section-title-group">
                                        <h3>Payroll & Compensation</h3>
                                        <p>Salary structure, allowances, and deductions</p>
                                    </div>
                                    <span class="section-badge success">New</span>
                                </div>

                                <!-- Salary Info Card -->
                                <div class="payroll-subsection">
                                    <h4 class="subsection-title">
                                        <span class="dashicons dashicons-bank"></span>
                                        Base Salary
                                    </h4>
                                    <div class="form-grid-modern">
                                        <div class="form-group-modern">
                                            <label class="form-label-modern">Basic Salary (Monthly)</label>
                                            <div class="input-wrapper-modern currency-input">
                                                <span class="currency-symbol"><?php echo esc_html(get_option('hrm_payroll_settings', ['currency' => '$'])['currency'] ?? '$'); ?></span>
                                                <input type="number" name="basic_salary" step="0.01" min="0" placeholder="0.00" class="input-modern input-currency">
                                            </div>
                                            <span class="field-hint">Base monthly salary before allowances</span>
                                        </div>
                                        
                                        <div class="form-group-modern">
                                            <label class="form-label-modern">Overtime Rate (Per Hour)</label>
                                            <div class="input-wrapper-modern currency-input">
                                                <span class="currency-symbol"><?php echo esc_html(get_option('hrm_payroll_settings', ['currency' => '$'])['currency'] ?? '$'); ?></span>
                                                <input type="number" name="overtime_rate" step="0.01" min="0" placeholder="0.00" class="input-modern input-currency">
                                            </div>
                                            <span class="field-hint">Hourly rate for overtime work</span>
                                        </div>
                                    </div>
                                </div>

                                <!-- Allowances Section -->
                                <div class="payroll-subsection">
                                    <div class="subsection-header">
                                        <h4 class="subsection-title">
                                            <span class="dashicons dashicons-plus-alt"></span>
                                            Allowances
                                            <span class="item-count" id="allowance-count">0</span>
                                        </h4>
                                        <button type="button" class="hrm-btn-add-item" id="add-allowance-btn">
                                            <span class="dashicons dashicons-plus"></span>
                                            Add Allowance
                                        </button>
                                    </div>
                                    <p class="subsection-desc">Add recurring monthly allowances like HRA, Transport, Medical, etc.</p>
                                    
                                    <div id="allowances-container" class="dynamic-items-container">
                                        <!-- Dynamic allowance items will be added here -->
                                        <div class="empty-state-mini" id="no-allowances">
                                            <span class="dashicons dashicons-plus-alt2"></span>
                                            <p>No allowances added yet</p>
                                        </div>
                                    </div>
                                </div>

                                <!-- Deductions Section -->
                                <div class="payroll-subsection">
                                    <div class="subsection-header">
                                        <h4 class="subsection-title">
                                            <span class="dashicons dashicons-minus"></span>
                                            Deductions
                                            <span class="item-count" id="deduction-count">0</span>
                                        </h4>
                                        <button type="button" class="hrm-btn-add-item" id="add-deduction-btn">
                                            <span class="dashicons dashicons-plus"></span>
                                            Add Deduction
                                        </button>
                                    </div>
                                    <p class="subsection-desc">Add fixed monthly deductions like Provident Fund, Insurance, Loans, etc.</p>
                                    
                                    <div id="deductions-container" class="dynamic-items-container">
                                        <!-- Dynamic deduction items will be added here -->
                                        <div class="empty-state-mini" id="no-deductions">
                                            <span class="dashicons dashicons-minus"></span>
                                            <p>No deductions added yet</p>
                                        </div>
                                    </div>
                                </div>

                                <!-- Bonuses Section -->
                                <div class="payroll-subsection">
                                    <div class="subsection-header">
                                        <h4 class="subsection-title">
                                            <span class="dashicons dashicons-awards"></span>
                                            Bonuses
                                            <span class="item-count" id="bonus-count">0</span>
                                        </h4>
                                        <button type="button" class="hrm-btn-add-item" id="add-bonus-btn">
                                            <span class="dashicons dashicons-plus"></span>
                                            Add Bonus
                                        </button>
                                    </div>
                                    <p class="subsection-desc">Define recurring bonuses like Performance Bonus, Annual Bonus, etc.</p>
                                    
                                    <div id="bonuses-container" class="dynamic-items-container">
                                        <!-- Dynamic bonus items will be added here -->
                                        <div class="empty-state-mini" id="no-bonuses">
                                            <span class="dashicons dashicons-awards"></span>
                                            <p>No bonuses added yet</p>
                                        </div>
                                    </div>
                                </div>

                                <!-- Tax Configuration -->
                                <div class="payroll-subsection tax-section">
                                    <h4 class="subsection-title">
                                        <span class="dashicons dashicons-chart-pie"></span>
                                        Statutory Deductions
                                    </h4>
                                    <div class="form-grid-modern">
                                        <div class="form-group-modern">
                                            <label class="form-label-modern">Tax Treatment</label>
                                            <select name="tax_treatment" class="select-modern">
                                                <option value="standard">Standard (Use Global Tax Slabs)</option>
                                                <option value="exempt">Tax Exempt</option>
                                                <option value="custom">Custom Tax Rate</option>
                                            </select>
                                            <span class="field-hint">How to calculate income tax for this employee</span>
                                        </div>
                                        
                                        <div class="form-group-modern">
                                            <label class="form-label-modern">Custom Tax Rate (%)</label>
                                            <div class="input-wrapper-modern">
                                                <input type="number" name="custom_tax_rate" step="0.01" min="0" max="100" placeholder="0.00" class="input-modern" disabled id="custom-tax-input">
                                                <span class="input-suffix">%</span>
                                            </div>
                                            <span class="field-hint">Only applicable if custom rate selected</span>
                                        </div>
                                    </div>
                                </div>

                                <!-- Salary Preview Card -->
                                <div class="salary-preview-card">
                                    <div class="preview-card-header">
                                        <span class="dashicons dashicons-chart-area"></span>
                                        <h4>Monthly Salary Preview</h4>
                                    </div>
                                    <div class="salary-breakdown">
                                        <div class="breakdown-row">
                                            <span class="breakdown-label">Basic Salary</span>
                                            <span class="breakdown-value" id="preview-basic">$0.00</span>
                                        </div>
                                        <div class="breakdown-row positive">
                                            <span class="breakdown-label">+ Total Allowances</span>
                                            <span class="breakdown-value" id="preview-allowances">$0.00</span>
                                        </div>
                                        <div class="breakdown-row positive">
                                            <span class="breakdown-label">+ Total Bonuses</span>
                                            <span class="breakdown-value" id="preview-bonuses">$0.00</span>
                                        </div>
                                        <div class="breakdown-divider"></div>
                                        <div class="breakdown-row gross">
                                            <span class="breakdown-label">Gross Salary</span>
                                            <span class="breakdown-value" id="preview-gross">$0.00</span>
                                        </div>
                                        <div class="breakdown-row negative">
                                            <span class="breakdown-label">- Total Deductions</span>
                                            <span class="breakdown-value" id="preview-deductions">$0.00</span>
                                        </div>
                                        <div class="breakdown-divider"></div>
                                        <div class="breakdown-row net">
                                            <span class="breakdown-label">Estimated Net Salary</span>
                                            <span class="breakdown-value" id="preview-net">$0.00</span>
                                        </div>
                                    </div>
                                    <p class="preview-note">* This is an estimate. Actual salary may vary based on attendance, overtime, and taxes.</p>
                                </div>
                            </section>

                            <!-- Section 4: Leave & Probation -->
                            <section class="hrm-form-section-modern" data-section="4">
                                <div class="section-header-modern">
                                    <div class="section-icon-box leave">
                                        <span class="dashicons dashicons-clock"></span>
                                    </div>
                                    <div class="section-title-group">
                                        <h3>Leave & Probation</h3>
                                        <p>Leave quota and probation period settings</p>
                                    </div>
                                </div>
                                
                                <div class="form-grid-modern three-col">
                                    <div class="form-group-modern">
                                        <label class="form-label-modern">Annual Leaves Allowed</label>
                                        <div class="input-wrapper-modern">
                                            <span class="input-prefix-icon dashicons dashicons-calendar"></span>
                                            <input type="number" name="leaves_allowed" min="0" max="365" step="1" value="20" class="input-modern">
                                            <span class="input-suffix">days</span>
                                        </div>
                                        <span class="field-hint">Total annual leave quota</span>
                                    </div>
                                    
                                    <div class="form-group-modern">
                                        <label class="form-label-modern">Probation Period</label>
                                        <div class="input-wrapper-modern">
                                            <span class="input-prefix-icon dashicons dashicons-backup"></span>
                                            <input type="number" name="probation_period" min="0" max="24" step="1" value="3" class="input-modern">
                                            <span class="input-suffix">months</span>
                                        </div>
                                        <span class="field-hint">Duration of probation</span>
                                    </div>
                                    
                                    <div class="form-group-modern">
                                        <label class="form-label-modern">Leave Applicability</label>
                                        <select name="leave_applicability" class="select-modern">
                                            <option value="after_probation">After Probation</option>
                                            <option value="during_probation">During Probation</option>
                                        </select>
                                        <span class="field-hint">When leaves can be availed</span>
                                    </div>
                                </div>
                            </section>

                            <!-- Section 5: Security -->
                            <section class="hrm-form-section-modern" data-section="5">
                                <div class="section-header-modern">
                                    <div class="section-icon-box security">
                                        <span class="dashicons dashicons-shield"></span>
                                    </div>
                                    <div class="section-title-group">
                                        <h3>Security Credentials</h3>
                                        <p>Set login password for the employee portal</p>
                                    </div>
                                    <span class="section-badge warning">Secure</span>
                                </div>
                                
                                <div class="form-grid-modern">
                                    <div class="form-group-modern full-width">
                                        <label class="form-label-modern">
                                            Password <span class="required">*</span>
                                        </label>
                                        <div class="password-input-modern">
                                            <span class="input-prefix-icon dashicons dashicons-lock"></span>
                                            <input type="password" name="password" id="password-input" required placeholder="Enter a strong password" minlength="8" class="input-modern">
                                            <button type="button" class="toggle-password-modern" title="Show password">
                                                <span class="dashicons dashicons-visibility"></span>
                                            </button>
                                        </div>
                                        <div class="password-strength-modern">
                                            <div class="strength-bar-container">
                                                <div class="strength-bar" id="strength-bar"></div>
                                            </div>
                                            <span class="strength-text" id="strength-text">Enter password</span>
                                        </div>
                                        <div class="password-requirements">
                                            <span class="req-item" data-req="length"><span class="dashicons dashicons-yes"></span> 8+ characters</span>
                                            <span class="req-item" data-req="lower"><span class="dashicons dashicons-yes"></span> Lowercase</span>
                                            <span class="req-item" data-req="upper"><span class="dashicons dashicons-yes"></span> Uppercase</span>
                                            <span class="req-item" data-req="number"><span class="dashicons dashicons-yes"></span> Number</span>
                                        </div>
                                    </div>
                                </div>
                            </section>
                        </main>
                    </div>

                    <!-- Form Footer -->
                    <footer class="hrm-form-footer-modern">
                        <div class="footer-left">
                            <a href="<?php echo admin_url('admin.php?page=hrm-employees'); ?>" class="hrm-btn-cancel">
                                <span class="dashicons dashicons-no-alt"></span>
                                <span>Cancel</span>
                            </a>
                        </div>
                        <div class="footer-right">
                            <button type="reset" class="hrm-btn-reset">
                                <span class="dashicons dashicons-image-rotate"></span>
                                <span>Reset Form</span>
                            </button>
                            <button type="submit" class="hrm-btn-submit" id="submit-btn">
                                <span class="btn-content">
                                    <span class="dashicons dashicons-plus-alt2"></span>
                                    <span>Create Employee</span>
                                </span>
                                <span class="btn-loading" style="display:none;">
                                    <span class="loading-spinner"></span>
                                    <span>Creating...</span>
                                </span>
                            </button>
                        </div>
                    </footer>
                </form>
            </div>
        </div>

        <!-- Success/Error Message Modal -->
        <div id="hrm-message-modal" class="hrm-modal" style="display:none;">
            <div class="hrm-modal-content hrm-premium-modal" style="max-width: 420px; text-align: center;">
                <div class="hrm-modal-header-premium" style="justify-content: center; border-bottom: none; padding-bottom: 0;">
                    <div class="header-icon-wrapper" id="msg-icon-wrapper" style="margin: 0 auto; width: 72px; height: 72px; display: flex; align-items: center; justify-content: center; border-radius: 50%;">
                        <span class="dashicons" id="msg-icon" style="font-size: 36px; width: 36px; height: 36px;"></span>
                    </div>
                </div>
                <div class="hrm-modal-body" style="padding: 16px 32px 32px;">
                    <h2 id="msg-title" style="font-size: 22px; font-weight: 700; color: #111827; margin-bottom: 12px;">Title</h2>
                    <p id="msg-text" style="color: #6B7280; font-size: 15px; margin-bottom: 28px; line-height: 1.6;">Message goes here.</p>
                    <button type="button" class="hrm-btn-primary" onclick="jQuery('#hrm-message-modal').fadeOut();" style="width: 100%; justify-content: center; padding: 14px 24px;">OK</button>
                </div>
            </div>
        </div>

        <script>
        jQuery(document).ready(function($) {
            const currencySymbol = '<?php echo esc_js(get_option('hrm_payroll_settings', ['currency' => '$'])['currency'] ?? '$'); ?>';
            
            // ========================
            // Profile Picture Handling
            // ========================
            var profilePicInput = $('#profile-picture-input');
            var previewContainer = $('#profile-preview');
            var uploadBtn = $('#upload-photo-btn');
            var removeBtn = $('#remove-photo-btn');
            
            uploadBtn.on('click', function() {
                profilePicInput.click();
            });
            
            // Also click on preview to upload
            previewContainer.on('click', function() {
                profilePicInput.click();
            });
            
            profilePicInput.on('change', function() {
                var file = this.files[0];
                if (file) {
                    var allowedTypes = ['image/jpeg', 'image/png', 'image/gif', 'image/webp'];
                    if (!allowedTypes.includes(file.type)) {
                        hrmToast('error', 'Please upload a valid image file (JPG, PNG, GIF, or WebP)');
                        this.value = '';
                        return;
                    }
                    
                    if (file.size > 2 * 1024 * 1024) {
                        hrmToast('error', 'File size must be less than 2MB');
                        this.value = '';
                        return;
                    }
                    
                    var reader = new FileReader();
                    reader.onload = function(e) {
                        previewContainer.html('<img src="' + e.target.result + '" alt="Profile Preview"><div class="preview-overlay"><span class="dashicons dashicons-camera-alt"></span></div>');
                        previewContainer.addClass('has-image');
                        removeBtn.show();
                        uploadBtn.find('.btn-text').text('Change Photo');
                        // Update mini preview
                        $('#preview-avatar-mini').html('<img src="' + e.target.result + '" alt="Preview">');
                    }
                    reader.readAsDataURL(file);
                }
            });
            
            removeBtn.on('click', function(e) {
                e.stopPropagation();
                profilePicInput.val('');
                previewContainer.html('<span class="dashicons dashicons-admin-users"></span><div class="preview-overlay"><span class="dashicons dashicons-camera-alt"></span></div>');
                previewContainer.removeClass('has-image');
                $(this).hide();
                uploadBtn.find('.btn-text').text('Upload Photo');
                $('#preview-avatar-mini').html('<span class="dashicons dashicons-admin-users"></span>');
            });

            // ========================
            // Live Preview Updates
            // ========================
            $('#input-name').on('input', function() {
                var name = $(this).val() || 'Employee Name';
                $('#preview-name').text(name);
            });
            
            $('#input-position, #input-department').on('input', function() {
                var position = $('#input-position').val() || 'Position';
                var department = $('#input-department').val() || 'Department';
                $('#preview-role').text(position + ' • ' + department);
            });

            // ========================
            // Password Handling
            // ========================
            $('.toggle-password-modern').on('click', function() {
                var input = $('#password-input');
                var icon = $(this).find('.dashicons');
                if (input.attr('type') === 'password') {
                    input.attr('type', 'text');
                    icon.removeClass('dashicons-visibility').addClass('dashicons-hidden');
                } else {
                    input.attr('type', 'password');
                    icon.removeClass('dashicons-hidden').addClass('dashicons-visibility');
                }
            });
            
            $('#password-input').on('input', function() {
                var password = $(this).val();
                var strength = 0;
                var bar = $('#strength-bar');
                var text = $('#strength-text');
                var reqs = {
                    length: password.length >= 8,
                    lower: /[a-z]/.test(password),
                    upper: /[A-Z]/.test(password),
                    number: /[0-9]/.test(password)
                };
                
                // Update requirement indicators
                $.each(reqs, function(key, met) {
                    var $req = $('.req-item[data-req="' + key + '"]');
                    if (met) {
                        $req.addClass('met');
                        strength += 25;
                    } else {
                        $req.removeClass('met');
                    }
                });
                
                bar.css('width', strength + '%');
                bar.removeClass('weak medium strong very-strong');
                
                if (password.length === 0) {
                    text.text('Enter password');
                    bar.css('width', '0%');
                } else if (strength <= 25) {
                    bar.addClass('weak');
                    text.text('Weak - needs improvement');
                } else if (strength <= 50) {
                    bar.addClass('medium');
                    text.text('Fair - add more variety');
                } else if (strength <= 75) {
                    bar.addClass('strong');
                    text.text('Good - almost there');
                } else {
                    bar.addClass('very-strong');
                    text.text('Strong password');
                }
            });

            // ========================
            // Payroll Section
            // ========================
            
            // Make payroll subsections collapsible
            $('.payroll-subsection .subsection-header').on('click', function() {
                var $parent = $(this).closest('.payroll-subsection');
                $parent.toggleClass('collapsed');
                $parent.find('.dynamic-items-container, .form-grid-modern, .subsection-desc').slideToggle(200);
            });
            
            // Tax treatment toggle
            $('select[name="tax_treatment"]').on('change', function() {
                var val = $(this).val();
                if (val === 'custom') {
                    $('#custom-tax-input').prop('disabled', false).focus();
                } else {
                    $('#custom-tax-input').prop('disabled', true).val('');
                }
            });
            
            // Dynamic Item Creator
            function createDynamicItem(type, label = '', value = '') {
                const itemId = Date.now() + Math.random().toString(36).substr(2, 9);
                return `
                <div class="dynamic-item" data-type="${type}" data-id="${itemId}">
                    <div class="item-drag-handle">
                        <span class="dashicons dashicons-menu"></span>
                    </div>
                    <div class="item-fields">
                        <input type="text" name="${type}_labels[]" class="input-modern item-label" placeholder="Label (e.g. ${type === 'allowance' ? 'HRA, Transport' : type === 'deduction' ? 'PF, Insurance' : 'Performance Bonus'})" value="${label}" required>
                        <div class="input-wrapper-modern currency-input item-value-wrap">
                            <span class="currency-symbol">${currencySymbol}</span>
                            <input type="number" name="${type}_values[]" class="input-modern input-currency item-value" step="0.01" min="0" placeholder="0.00" value="${value}" required>
                        </div>
                    </div>
                    <button type="button" class="btn-remove-item" title="Remove">
                        <span class="dashicons dashicons-trash"></span>
                    </button>
                </div>`;
            }
            
            // Add Allowance
            $('#add-allowance-btn').on('click', function() {
                $('#no-allowances').hide();
                $('#allowances-container').append(createDynamicItem('allowance'));
                updateItemCounts();
                updateSalaryPreview();
            });
            
            // Add Deduction
            $('#add-deduction-btn').on('click', function() {
                $('#no-deductions').hide();
                $('#deductions-container').append(createDynamicItem('deduction'));
                updateItemCounts();
                updateSalaryPreview();
            });
            
            // Add Bonus
            $('#add-bonus-btn').on('click', function() {
                $('#no-bonuses').hide();
                $('#bonuses-container').append(createDynamicItem('bonus'));
                updateItemCounts();
                updateSalaryPreview();
            });
            
            // Remove Item
            $(document).on('click', '.btn-remove-item', function() {
                var container = $(this).closest('.dynamic-items-container');
                var type = $(this).closest('.dynamic-item').data('type');
                $(this).closest('.dynamic-item').fadeOut(200, function() {
                    $(this).remove();
                    updateItemCounts();
                    updateSalaryPreview();
                    // Show empty state if no items
                    if (container.find('.dynamic-item').length === 0) {
                        container.find('.empty-state-mini').show();
                    }
                });
            });
            
            // Update Item counts
            function updateItemCounts() {
                $('#allowance-count').text($('#allowances-container .dynamic-item').length);
                $('#deduction-count').text($('#deductions-container .dynamic-item').length);
                $('#bonus-count').text($('#bonuses-container .dynamic-item').length);
            }
            
            // Update Salary Preview
            function updateSalaryPreview() {
                var basic = parseFloat($('input[name="basic_salary"]').val()) || 0;
                var totalAllowances = 0;
                var totalDeductions = 0;
                var totalBonuses = 0;
                
                $('#allowances-container .dynamic-item .item-value').each(function() {
                    totalAllowances += parseFloat($(this).val()) || 0;
                });
                
                $('#deductions-container .dynamic-item .item-value').each(function() {
                    totalDeductions += parseFloat($(this).val()) || 0;
                });
                
                $('#bonuses-container .dynamic-item .item-value').each(function() {
                    totalBonuses += parseFloat($(this).val()) || 0;
                });
                
                var gross = basic + totalAllowances + totalBonuses;
                var net = gross - totalDeductions;
                
                $('#preview-basic').text(currencySymbol + formatNumber(basic));
                $('#preview-allowances').text(currencySymbol + formatNumber(totalAllowances));
                $('#preview-bonuses').text(currencySymbol + formatNumber(totalBonuses));
                $('#preview-gross').text(currencySymbol + formatNumber(gross));
                $('#preview-deductions').text(currencySymbol + formatNumber(totalDeductions));
                $('#preview-net').text(currencySymbol + formatNumber(net));
            }
            
            function formatNumber(num) {
                return num.toLocaleString('en-US', { minimumFractionDigits: 2, maximumFractionDigits: 2 });
            }
            
            // Listen for value changes
            $(document).on('input', 'input[name="basic_salary"], .dynamic-item .item-value', function() {
                updateSalaryPreview();
            });

            // ========================
            // Section Progress Tracking
            // ========================
            function updateProgress() {
                var sections = $('.hrm-form-section-modern');
                sections.each(function(index) {
                    var $section = $(this);
                    var $step = $('.progress-step[data-step="' + (index + 1) + '"]');
                    var hasContent = false;
                    
                    $section.find('input[required]').each(function() {
                        if ($(this).val().trim() !== '') {
                            hasContent = true;
                        }
                    });
                    
                    if (hasContent) {
                        $step.addClass('completed');
                    } else {
                        $step.removeClass('completed');
                    }
                });
            }
            
            $(document).on('input', 'input[required]', updateProgress);

            // ========================
            // Form Submission
            // ========================
            window.showHrmModal = function(type, title, message) {
                var modal = $('#hrm-message-modal');
                modal.find('#msg-title').text(title);
                modal.find('#msg-text').text(message);
                
                var iconWrapper = modal.find('#msg-icon-wrapper');
                var icon = modal.find('#msg-icon');
                
                icon.removeClass().addClass('dashicons');
                
                if(type === 'success') {
                    iconWrapper.css('background', 'linear-gradient(135deg, #D1FAE5 0%, #A7F3D0 100%)').css('color', '#065F46');
                    icon.addClass('dashicons-yes-alt');
                    modal.find('button').off('click').on('click', function() {
                        window.location.href = '<?php echo admin_url('admin.php?page=hrm-employees'); ?>';
                    });
                } else {
                    iconWrapper.css('background', 'linear-gradient(135deg, #FEE2E2 0%, #FECACA 100%)').css('color', '#991B1B');
                    icon.addClass('dashicons-warning');
                    if(title === 'User Already Exists') {
                        icon.removeClass('dashicons-warning').addClass('dashicons-admin-users');
                    }
                    modal.find('button').off('click').on('click', function() {
                        modal.fadeOut();
                    });
                }
                
                modal.css('display', 'flex').hide().fadeIn();
            };
            
            // Form Reset Handler
            $('#hrm-add-employee-form').on('reset', function() {
                setTimeout(function() {
                    // Reset previews
                    $('#preview-name').text('Employee Name');
                    $('#preview-role').text('Position • Department');
                    $('#preview-avatar-mini').html('<span class="dashicons dashicons-admin-users"></span>');
                    previewContainer.html('<span class="dashicons dashicons-admin-users"></span><div class="preview-overlay"><span class="dashicons dashicons-camera-alt"></span></div>').removeClass('has-image');
                    removeBtn.hide();
                    uploadBtn.find('.btn-text').text('Upload Photo');
                    
                    // Clear dynamic items
                    $('.dynamic-item').remove();
                    $('.empty-state-mini').show();
                    updateItemCounts();
                    updateSalaryPreview();
                    
                    // Reset password indicators
                    $('#strength-bar').css('width', '0%').removeClass('weak medium strong very-strong');
                    $('#strength-text').text('Enter password');
                    $('.req-item').removeClass('met');
                    
                    // Reset progress
                    $('.progress-step').removeClass('completed active');
                    $('.progress-step[data-step="1"]').addClass('active');
                }, 10);
            });
            
            // Toast function
            window.hrmToast = window.hrmToast || function(type, message) {
                var icon = type === 'success' ? 'dashicons-yes-alt' : 'dashicons-warning';
                var html = `
                <div class="hrm-toast-notification ${type}">
                    <span class="dashicons ${icon}"></span>
                    <span>${message}</span>
                    <button class="hrm-toast-close" onclick="this.parentElement.remove()">&times;</button>
                </div>`;
                
                $('body').append(html);
                setTimeout(function() {
                    $('.hrm-toast-notification').fadeOut(function() { $(this).remove(); });
                }, 4000);
            };
        });
        </script>
        <?php
    }

    public static function render_employee_profile() {
        if ( ! isset( $_GET['id'] ) ) return;
        $employee_id = absint( $_GET['id'] );
        $employee = get_post( $employee_id );
        if ( ! $employee || $employee->post_type !== HRM_Setup::EMPLOYEE_POST_TYPE ) return;

        $position = get_post_meta( $employee_id, 'hrm_position', true ) ?: 'Employee';
        $email = get_post_meta( $employee_id, 'hrm_email', true ) ?: 'N/A';
        $phone = get_post_meta( $employee_id, 'hrm_phone', true ) ?: 'N/A';
        $department = get_post_meta( $employee_id, 'hrm_department', true ) ?: 'General';
        $profile_pic_id = get_post_meta( $employee_id, 'hrm_profile_pic', true );
        $joining_date = get_post_meta( $employee_id, 'hrm_joining_date', true );

        global $wpdb;
        $total_tasks = $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(*) FROM {$wpdb->prefix}hrm_tasks WHERE assigned_to = %d", $employee_id ) );
        $completed_tasks = $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(*) FROM {$wpdb->prefix}hrm_tasks WHERE assigned_to = %d AND status = 'completed'", $employee_id ) );


        ?>
        <div class="wrap hrm-wrap hrm-modern-dashboard">
             <div class="hrm-header-section">
                <div class="hrm-header-left">
                     <a href="<?php echo admin_url('admin.php?page=hrm-employees'); ?>" class="hrm-back-link">← Back to Employees</a>
                    <h1>Employee Profile</h1>
                </div>
            </div>

            <div class="hrm-profile-grid">
                <!-- Sidebar -->
                <div class="hrm-profile-sidebar">
                    <div class="hrm-card hrm-profile-card-lg">
                        <div class="profile-avatar-lg">
                            <?php 
                            if ( $profile_pic_id ) {
                                echo wp_get_attachment_image( $profile_pic_id, 'thumbnail', false, ['class' => 'avatar-img'] );
                            } else {
                                echo strtoupper(substr($employee->post_title, 0, 1)); 
                            }
                            ?>
                        </div>
                        <h2><?php echo esc_html( $employee->post_title ); ?></h2>
                        <p class="profile-role"><?php echo esc_html( $position ); ?></p>
                        <span class="hrm-badge badge-blue"><?php echo esc_html( $department ); ?></span>

                        <div class="profile-contact-info">
                            <div class="contact-item">
                                <span class="dashicons dashicons-email"></span>
                                <span><?php echo esc_html( $email ); ?></span>
                            </div>
                            <div class="contact-item">
                                <span class="dashicons dashicons-phone"></span>
                                <span><?php echo esc_html( $phone ); ?></span>
                            </div>
                            <?php if ( $joining_date ) : ?>
                            <div class="contact-item">
                                <span class="dashicons dashicons-calendar-alt"></span>
                                <span>Joined: <?php echo esc_html( date('M j, Y', strtotime($joining_date)) ); ?></span>
                            </div>
                            <?php endif; ?>
                        </div>

                        <!-- Admin Actions -->
                        <div style="margin-top: 20px; border-top: 1px solid #eee; padding-top: 15px;">
                            <button class="hrm-btn-outline btn-sm" onclick="jQuery('#reset-pwd-modal').show()">Reset Password</button>
                            <?php 
                                $status = $employee->post_status;
                                if ($status === 'publish') {
                                    echo '<button class="hrm-btn-outline btn-sm hrm-toggle-status" data-id="'.$employee_id.'" data-status="draft" style="color:orange;">Deactivate Account</button>';
                                } else {
                                    echo '<button class="hrm-btn-outline btn-sm hrm-toggle-status" data-id="'.$employee_id.'" data-status="publish" style="color:green;">Activate Account</button>';
                                }
                            ?>
                        </div>
                    </div>
                </div>

                <!-- Main Content -->
                <div class="hrm-profile-content">
                    <?php
                    // Get salary structure
                    $salary_structure = $wpdb->get_row($wpdb->prepare(
                        "SELECT * FROM {$wpdb->prefix}hrm_salary_structures WHERE employee_id = %d",
                        $employee_id
                    ));
                    
                    // Get leave & probation data
                    $leaves_allowed = get_post_meta($employee_id, 'hrm_leaves_allowed', true) ?: 0;
                    $probation_period = get_post_meta($employee_id, 'hrm_probation_period', true) ?: 0;
                    $leave_applicability = get_post_meta($employee_id, 'hrm_leave_applicability', true) ?: 'after_probation';
                    
                    // Get payroll settings
                    $payroll_settings = get_option('hrm_payroll_settings', ['currency' => '$']);
                    $currency = $payroll_settings['currency'] ?? '$';
                    
                    // Get recent payroll
                    $recent_payroll = $wpdb->get_results($wpdb->prepare(
                        "SELECT * FROM {$wpdb->prefix}hrm_payroll WHERE employee_id = %d ORDER BY payroll_month DESC LIMIT 6",
                        $employee_id
                    ));
                    ?>
                    
                    <!-- Salary Structure Card -->
                    <div class="hrm-card hrm-profile-section-card">
                        <div class="profile-card-header">
                            <div class="header-icon-wrap payroll">
                                <span class="dashicons dashicons-money-alt"></span>
                            </div>
                            <div class="header-text">
                                <h3>Salary Structure</h3>
                                <p>Monthly compensation breakdown</p>
                            </div>
                            <a href="<?php echo admin_url('admin.php?page=hrm-payroll&tab=structures'); ?>" class="hrm-btn-outline btn-sm">
                                <span class="dashicons dashicons-edit"></span> Edit Structure
                            </a>
                        </div>
                        
                        <?php if ($salary_structure): ?>
                        <div class="salary-structure-grid">
                            <!-- Basic Salary -->
                            <div class="salary-item primary">
                                <span class="salary-label">Basic Salary</span>
                                <span class="salary-value"><?php echo esc_html($currency . number_format($salary_structure->basic_salary, 2)); ?></span>
                            </div>
                            
                            <!-- Overtime Rate -->
                            <div class="salary-item">
                                <span class="salary-label">Overtime Rate (per hour)</span>
                                <span class="salary-value"><?php echo esc_html($currency . number_format($salary_structure->overtime_rate, 2)); ?></span>
                            </div>
                            
                            <?php
                            $allowances = json_decode($salary_structure->allowances, true) ?: [];
                            $deductions = json_decode($salary_structure->deductions, true) ?: [];
                            $tax_rules = json_decode($salary_structure->tax_rules, true) ?: [];
                            $bonuses = $tax_rules['bonuses'] ?? [];
                            
                            $total_allowances = array_sum(array_column($allowances, 'value'));
                            $total_deductions = array_sum(array_column($deductions, 'value'));
                            $total_bonuses = array_sum(array_column($bonuses, 'value'));
                            ?>
                            
                            <!-- Allowances -->
                            <?php if (!empty($allowances)): ?>
                            <div class="salary-section allowances">
                                <div class="section-header-mini">
                                    <span class="dashicons dashicons-plus-alt"></span>
                                    <span>Allowances</span>
                                    <span class="section-total positive">+<?php echo esc_html($currency . number_format($total_allowances, 2)); ?></span>
                                </div>
                                <div class="section-items">
                                    <?php foreach ($allowances as $item): ?>
                                    <div class="item-row">
                                        <span class="item-label"><?php echo esc_html($item['label']); ?></span>
                                        <span class="item-value"><?php echo esc_html($currency . number_format($item['value'], 2)); ?></span>
                                    </div>
                                    <?php endforeach; ?>
                                </div>
                            </div>
                            <?php endif; ?>
                            
                            <!-- Bonuses -->
                            <?php if (!empty($bonuses)): ?>
                            <div class="salary-section bonuses">
                                <div class="section-header-mini">
                                    <span class="dashicons dashicons-awards"></span>
                                    <span>Bonuses</span>
                                    <span class="section-total positive">+<?php echo esc_html($currency . number_format($total_bonuses, 2)); ?></span>
                                </div>
                                <div class="section-items">
                                    <?php foreach ($bonuses as $item): ?>
                                    <div class="item-row">
                                        <span class="item-label"><?php echo esc_html($item['label']); ?></span>
                                        <span class="item-value"><?php echo esc_html($currency . number_format($item['value'], 2)); ?></span>
                                    </div>
                                    <?php endforeach; ?>
                                </div>
                            </div>
                            <?php endif; ?>
                            
                            <!-- Deductions -->
                            <?php if (!empty($deductions)): ?>
                            <div class="salary-section deductions">
                                <div class="section-header-mini">
                                    <span class="dashicons dashicons-minus"></span>
                                    <span>Deductions</span>
                                    <span class="section-total negative">-<?php echo esc_html($currency . number_format($total_deductions, 2)); ?></span>
                                </div>
                                <div class="section-items">
                                    <?php foreach ($deductions as $item): ?>
                                    <div class="item-row">
                                        <span class="item-label"><?php echo esc_html($item['label']); ?></span>
                                        <span class="item-value"><?php echo esc_html($currency . number_format($item['value'], 2)); ?></span>
                                    </div>
                                    <?php endforeach; ?>
                                </div>
                            </div>
                            <?php endif; ?>
                            
                            <!-- Net Salary Estimate -->
                            <div class="salary-summary">
                                <div class="summary-row gross">
                                    <span>Gross Salary</span>
                                    <span><?php echo esc_html($currency . number_format($salary_structure->basic_salary + $total_allowances + $total_bonuses, 2)); ?></span>
                                </div>
                                <div class="summary-row net">
                                    <span>Est. Net Salary</span>
                                    <span><?php echo esc_html($currency . number_format($salary_structure->basic_salary + $total_allowances + $total_bonuses - $total_deductions, 2)); ?></span>
                                </div>
                            </div>
                        </div>
                        <?php else: ?>
                        <div class="empty-state-card">
                            <span class="dashicons dashicons-money-alt"></span>
                            <p>No salary structure configured yet.</p>
                            <a href="<?php echo admin_url('admin.php?page=hrm-payroll&tab=structures'); ?>" class="hrm-btn-outline btn-sm">Configure Salary</a>
                        </div>
                        <?php endif; ?>
                    </div>
                    
                    <!-- Leave & Probation Card -->
                    <div class="hrm-card hrm-profile-section-card">
                        <div class="profile-card-header">
                            <div class="header-icon-wrap leave">
                                <span class="dashicons dashicons-calendar-alt"></span>
                            </div>
                            <div class="header-text">
                                <h3>Leave & Probation</h3>
                                <p>Leave quota and probation status</p>
                            </div>
                        </div>
                        
                        <div class="leave-probation-grid">
                            <div class="info-box">
                                <span class="info-icon"><span class="dashicons dashicons-calendar"></span></span>
                                <div class="info-content">
                                    <span class="info-value"><?php echo esc_html($leaves_allowed); ?></span>
                                    <span class="info-label">Annual Leaves</span>
                                </div>
                            </div>
                            
                            <div class="info-box">
                                <span class="info-icon"><span class="dashicons dashicons-backup"></span></span>
                                <div class="info-content">
                                    <span class="info-value"><?php echo esc_html($probation_period); ?> months</span>
                                    <span class="info-label">Probation Period</span>
                                </div>
                            </div>
                            
                            <div class="info-box">
                                <span class="info-icon"><span class="dashicons dashicons-clock"></span></span>
                                <div class="info-content">
                                    <span class="info-value"><?php echo $leave_applicability === 'during_probation' ? 'During Probation' : 'After Probation'; ?></span>
                                    <span class="info-label">Leave Applicability</span>
                                </div>
                            </div>
                            
                            <?php
                            // Calculate probation end date
                            if ($joining_date && $probation_period > 0) {
                                $probation_end = date('Y-m-d', strtotime($joining_date . ' + ' . $probation_period . ' months'));
                                $is_on_probation = strtotime($probation_end) > time();
                            ?>
                            <div class="info-box <?php echo $is_on_probation ? 'warning' : 'success'; ?>">
                                <span class="info-icon"><span class="dashicons dashicons-<?php echo $is_on_probation ? 'warning' : 'yes-alt'; ?>"></span></span>
                                <div class="info-content">
                                    <span class="info-value"><?php echo $is_on_probation ? 'On Probation' : 'Completed'; ?></span>
                                    <span class="info-label">Ends: <?php echo date('M j, Y', strtotime($probation_end)); ?></span>
                                </div>
                            </div>
                            <?php } ?>
                        </div>
                    </div>
                    
                    <!-- Recent Payroll History -->
                    <?php if (!empty($recent_payroll)): ?>
                    <div class="hrm-card hrm-profile-section-card">
                        <div class="profile-card-header">
                            <div class="header-icon-wrap history">
                                <span class="dashicons dashicons-list-view"></span>
                            </div>
                            <div class="header-text">
                                <h3>Recent Payroll</h3>
                                <p>Last 6 months payroll history</p>
                            </div>
                            <a href="<?php echo admin_url('admin.php?page=hrm-payroll&tab=history'); ?>" class="hrm-btn-outline btn-sm">
                                View All
                            </a>
                        </div>
                        
                        <div class="payroll-history-table">
                            <table class="hrm-table">
                                <thead>
                                    <tr>
                                        <th>Month</th>
                                        <th>Gross</th>
                                        <th>Deductions</th>
                                        <th>Net Salary</th>
                                        <th>Status</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($recent_payroll as $payroll): ?>
                                    <tr>
                                        <td><strong><?php echo date('M Y', strtotime($payroll->payroll_month . '-01')); ?></strong></td>
                                        <td><?php echo esc_html($currency . number_format($payroll->basic_salary + $payroll->total_allowances + $payroll->total_bonuses + $payroll->overtime_pay, 2)); ?></td>
                                        <td class="text-danger"><?php echo esc_html($currency . number_format($payroll->total_deductions + $payroll->attendance_penalty + $payroll->tax_amount, 2)); ?></td>
                                        <td class="text-success"><strong><?php echo esc_html($currency . number_format($payroll->net_salary, 2)); ?></strong></td>
                                        <td>
                                            <?php 
                                            $status_class = $payroll->status === 'paid' ? 'success' : ($payroll->status === 'approved' ? 'info' : 'warning');
                                            ?>
                                            <span class="hrm-badge badge-<?php echo $status_class; ?>"><?php echo ucfirst($payroll->status); ?></span>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                    <?php endif; ?>
                    
                </div>
            </div>
            
            <!-- Reset Password Modal -->
            <div id="reset-pwd-modal" class="hrm-modal" style="display:none;">
               <div class="hrm-modal-content">
                   <div class="hrm-modal-header">
                       <h2>Reset Password</h2>
                       <span class="hrm-close-modal" onclick="jQuery('#reset-pwd-modal').hide()">&times;</span>
                   </div>
                   <div class="hrm-modal-body">
                       <form id="hrm-reset-pwd-form">
                           <input type="hidden" name="action" value="hrm_reset_password">
                           <input type="hidden" name="employee_id" value="<?php echo $employee_id; ?>">
                           <div class="hrm-form-group">
                               <label>New Password</label>
                               <input type="text" name="password" required>
                           </div>
                           <button type="submit" class="hrm-btn-primary">Update Password</button>
                       </form>
                   </div>
               </div>
            </div>

            <script>
            jQuery(document).ready(function($) {
                // Toggle Status Handler
                $('.hrm-toggle-status').on('click', function() {
                    var btn = $(this);
                    var status = btn.data('status');
                    var confirmMsg = status === 'draft' ? 'Deactivate this account?' : 'Activate this account?';
                    
                    if(!confirm(confirmMsg)) return;
                    
                    var originalText = btn.text();
                    btn.prop('disabled', true).text('Processing...');
                    
                    $.post(ajaxurl, {
                        action: 'hrm_toggle_status',
                        employee_id: btn.data('id'),
                        status: status
                    }, function(res) {
                        if(res.success) {
                            location.reload();
                        } else {
                            alert('Error: ' + (res.data || 'Unknown error'));
                            btn.prop('disabled', false).text(originalText);
                        }
                    });
                });

                // Reset Password Handler
                $('#hrm-reset-pwd-form').on('submit', function(e) {
                    e.preventDefault();
                    var form = $(this);
                    var btn = form.find('button[type="submit"]');
                    var originalText = btn.text();
                    
                    btn.prop('disabled', true).text('Updating...');
                    
                    $.post(ajaxurl, form.serialize(), function(res) {
                        alert(res.success ? res.data : 'Error: ' + res.data);
                        if(res.success) {
                            $('#reset-pwd-modal').hide();
                            form[0].reset();
                        }
                        btn.prop('disabled', false).text(originalText);
                    });
                });
            });
            </script>
        </div>
        <?php
    }

    // --- AJAX Handlers ---

    public static function handle_ajax_add_employee() {
        check_ajax_referer( 'hrm_ajax_nonce', 'nonce' );
        if ( ! current_user_can( 'manage_options' ) ) wp_send_json_error( 'Unauthorized' );

        $name = sanitize_text_field( $_POST['name'] );
        $username = sanitize_user( $_POST['username'] );
        $email = sanitize_email( $_POST['email'] );
        $password = $_POST['password'];

        // Step 1: Check if username exists in wp_users
        $user_id = username_exists( $username );
        
        // Step 2: Check if email exists (fallback)
        if ( ! $user_id && email_exists( $email ) ) {
            $user_id = email_exists( $email );
        }

        $is_recovery = false;

        if ( $user_id ) {
            // User exists in wp_users - determine if orphaned or linked
            $existing_employee = get_posts([
                'post_type' => HRM_Setup::EMPLOYEE_POST_TYPE,
                'author' => $user_id,
                'posts_per_page' => 1,
                'post_status' => 'any' // Check all: publish, draft, trash
            ]);

            if ( ! empty($existing_employee) ) {
                // Employee profile exists - check status
                $status = $existing_employee[0]->post_status;
                
                if ( $status === 'trash' ) {
                    // Trashed profile - instruct user to restore or permanently delete
                    wp_send_json_error( 
                        "A trashed employee profile exists for this user. " .
                        "Please go to Employees → Trash, then either restore or permanently delete it before creating a new one."
                    );
                }
                
                // Active profile exists
                wp_send_json_error( 
                    "This username is already registered as: " . $existing_employee[0]->post_title . 
                    " (Status: " . ucfirst($status) . ")"
                );
            }
            
            // ORPHAN DETECTED: User exists but no employee profile
            // Auto-recover by linking to new profile
            $is_recovery = true;
            $user = get_userdata( $user_id );
            
            // Ensure hrm_employee role
            if ( ! in_array( 'hrm_employee', (array) $user->roles ) ) {
                $user->add_role( 'hrm_employee' );
            }
            
            // Update password to new one
            wp_set_password( $password, $user_id );
            
        } else {
            // No existing user - create new
            $user_id = wp_create_user( $username, $password, $email );
            
            if ( is_wp_error( $user_id ) ) {
                wp_send_json_error( $user_id->get_error_message() );
            }
            
            $user = new WP_User( $user_id );
            $user->set_role( 'hrm_employee' );
        }

        // Update display name
        wp_update_user([
            'ID' => $user_id,
            'display_name' => $name
        ]);

        // Create employee profile post
        $employee_id = wp_insert_post([
            'post_title'  => $name,
            'post_type'   => HRM_Setup::EMPLOYEE_POST_TYPE,
            'post_status' => 'publish',
            'post_author' => $user_id
        ]);

        if ( ! $employee_id || is_wp_error( $employee_id ) ) {
            wp_send_json_error( 'Failed to create employee profile.' );
        }

        // Save employee metadata
        update_post_meta( $employee_id, 'hrm_position', sanitize_text_field( $_POST['position'] ?? '' ) );
        update_post_meta( $employee_id, 'hrm_department', sanitize_text_field( $_POST['department'] ?? '' ) );
        update_post_meta( $employee_id, 'hrm_phone', sanitize_text_field( $_POST['phone'] ?? '' ) );
        update_post_meta( $employee_id, 'hrm_email', $email );
        
        // Save Joining Date
        if ( ! empty( $_POST['joining_date'] ) ) {
            $joining_date = sanitize_text_field( $_POST['joining_date'] );
            // Validate date format (YYYY-MM-DD)
            if ( preg_match('/^\d{4}-\d{2}-\d{2}$/', $joining_date) ) {
                update_post_meta( $employee_id, 'hrm_joining_date', $joining_date );
            }
        }
        
        // Save Leave & Probation Fields
        $leaves_allowed = isset( $_POST['leaves_allowed'] ) ? absint( $_POST['leaves_allowed'] ) : 0;
        $probation_period = isset( $_POST['probation_period'] ) ? absint( $_POST['probation_period'] ) : 0;
        $leave_applicability = isset( $_POST['leave_applicability'] ) ? sanitize_key( $_POST['leave_applicability'] ) : 'after_probation';
        
        // Validate leave_applicability value
        if ( ! in_array( $leave_applicability, [ 'during_probation', 'after_probation' ] ) ) {
            $leave_applicability = 'after_probation';
        }
        
        // Validate ranges
        $leaves_allowed = min( 365, max( 0, $leaves_allowed ) );
        $probation_period = min( 24, max( 0, $probation_period ) );
        
        update_post_meta( $employee_id, 'hrm_leaves_allowed', $leaves_allowed );
        update_post_meta( $employee_id, 'hrm_probation_period', $probation_period );
        update_post_meta( $employee_id, 'hrm_leave_applicability', $leave_applicability );
        
        // Handle Profile Picture Upload
        if ( ! empty( $_FILES['profile_picture']['name'] ) ) {
            $attachment_id = self::handle_profile_picture_upload( $employee_id, $_FILES['profile_picture'] );
            if ( $attachment_id && ! is_wp_error( $attachment_id ) ) {
                update_post_meta( $employee_id, 'hrm_profile_pic', $attachment_id );
            }
        }

        // Save Payroll / Salary Structure
        global $wpdb;
        
        $basic_salary = isset( $_POST['basic_salary'] ) ? floatval( $_POST['basic_salary'] ) : 0;
        $overtime_rate = isset( $_POST['overtime_rate'] ) ? floatval( $_POST['overtime_rate'] ) : 0;
        $tax_treatment = isset( $_POST['tax_treatment'] ) ? sanitize_key( $_POST['tax_treatment'] ) : 'standard';
        $custom_tax_rate = isset( $_POST['custom_tax_rate'] ) ? floatval( $_POST['custom_tax_rate'] ) : 0;
        
        // Build allowances array
        $allowances = [];
        if ( ! empty( $_POST['allowance_labels'] ) && is_array( $_POST['allowance_labels'] ) ) {
            foreach ( $_POST['allowance_labels'] as $i => $label ) {
                $label = sanitize_text_field( $label );
                $value = isset( $_POST['allowance_values'][$i] ) ? floatval( $_POST['allowance_values'][$i] ) : 0;
                if ( $label && $value >= 0 ) {
                    $allowances[] = [ 'label' => $label, 'value' => $value ];
                }
            }
        }
        
        // Build deductions array
        $deductions = [];
        if ( ! empty( $_POST['deduction_labels'] ) && is_array( $_POST['deduction_labels'] ) ) {
            foreach ( $_POST['deduction_labels'] as $i => $label ) {
                $label = sanitize_text_field( $label );
                $value = isset( $_POST['deduction_values'][$i] ) ? floatval( $_POST['deduction_values'][$i] ) : 0;
                if ( $label && $value >= 0 ) {
                    $deductions[] = [ 'label' => $label, 'value' => $value ];
                }
            }
        }
        
        // Build bonuses array
        $bonuses = [];
        if ( ! empty( $_POST['bonus_labels'] ) && is_array( $_POST['bonus_labels'] ) ) {
            foreach ( $_POST['bonus_labels'] as $i => $label ) {
                $label = sanitize_text_field( $label );
                $value = isset( $_POST['bonus_values'][$i] ) ? floatval( $_POST['bonus_values'][$i] ) : 0;
                if ( $label && $value >= 0 ) {
                    $bonuses[] = [ 'label' => $label, 'value' => $value ];
                }
            }
        }
        
        // Only create salary structure if there's payroll data
        if ( $basic_salary > 0 || ! empty( $allowances ) || ! empty( $deductions ) || ! empty( $bonuses ) ) {
            // Tax rules configuration
            $tax_rules = [
                'treatment' => $tax_treatment,
                'custom_rate' => $custom_tax_rate
            ];
            
            $salary_data = [
                'employee_id' => $employee_id,
                'basic_salary' => $basic_salary,
                'overtime_rate' => $overtime_rate,
                'allowances' => json_encode( $allowances ),
                'deductions' => json_encode( $deductions ),
                'tax_rules' => json_encode( array_merge( $tax_rules, [ 'bonuses' => $bonuses ] ) ),
                'created_at' => current_time( 'mysql' ),
                'updated_at' => current_time( 'mysql' )
            ];
            
            $wpdb->insert( "{$wpdb->prefix}hrm_salary_structures", $salary_data );
        }

        // Success message
        if ( $is_recovery ) {
            wp_send_json_success( 'Employee profile created by recovering orphaned user account.' );
        } else {
            wp_send_json_success( 'Employee profile created successfully.' );
        }
    }
    
    /**
     * Securely handle profile picture upload with validation
     */
    private static function handle_profile_picture_upload( $employee_id, $file ) {
        // Allowed mime types
        $allowed_types = [
            'image/jpeg' => 'jpg',
            'image/png'  => 'png',
            'image/gif'  => 'gif',
            'image/webp' => 'webp'
        ];
        
        // Validate file type
        $file_type = wp_check_filetype_and_ext( $file['tmp_name'], $file['name'] );
        if ( ! isset( $allowed_types[ $file_type['type'] ] ) ) {
            return new WP_Error( 'invalid_file_type', 'Invalid file type. Only JPG, PNG, GIF, and WebP are allowed.' );
        }
        
        // Validate file size (max 2MB)
        if ( $file['size'] > 2 * 1024 * 1024 ) {
            return new WP_Error( 'file_too_large', 'File size exceeds 2MB limit.' );
        }
        
        // Include required files
        require_once( ABSPATH . 'wp-admin/includes/image.php' );
        require_once( ABSPATH . 'wp-admin/includes/file.php' );
        require_once( ABSPATH . 'wp-admin/includes/media.php' );
        
        // Handle the upload
        $attachment_id = media_handle_upload( 'profile_picture', $employee_id );
        
        return $attachment_id;
    }

    public static function handle_ajax_edit_employee() {
        check_ajax_referer( 'hrm_ajax_nonce', 'nonce' );
        if ( ! current_user_can( 'manage_options' ) ) wp_send_json_error( 'Unauthorized' );

        $employee_id = absint( $_POST['employee_id'] );
        $name = sanitize_text_field( $_POST['name'] );
        
        // Verify employee exists
        $employee = get_post( $employee_id );
        if ( ! $employee || $employee->post_type !== HRM_Setup::EMPLOYEE_POST_TYPE ) {
            wp_send_json_error( 'Employee not found.' );
        }
        
        wp_update_post([ 'ID' => $employee_id, 'post_title' => $name ]);
        update_post_meta( $employee_id, 'hrm_position', sanitize_text_field( $_POST['position'] ?? '' ) );
        update_post_meta( $employee_id, 'hrm_department', sanitize_text_field( $_POST['department'] ?? '' ) );
        update_post_meta( $employee_id, 'hrm_phone', sanitize_text_field( $_POST['phone'] ?? '' ) );
        update_post_meta( $employee_id, 'hrm_email', sanitize_email( $_POST['email'] ?? '' ) );
        
        // Save Joining Date
        if ( isset( $_POST['joining_date'] ) ) {
            $joining_date = sanitize_text_field( $_POST['joining_date'] );
            if ( empty( $joining_date ) || preg_match('/^\d{4}-\d{2}-\d{2}$/', $joining_date) ) {
                update_post_meta( $employee_id, 'hrm_joining_date', $joining_date );
            }
        }
        
        // Save Leave & Probation Fields
        if ( isset( $_POST['leaves_allowed'] ) ) {
            $leaves_allowed = absint( $_POST['leaves_allowed'] );
            $leaves_allowed = min( 365, max( 0, $leaves_allowed ) );
            update_post_meta( $employee_id, 'hrm_leaves_allowed', $leaves_allowed );
        }
        
        if ( isset( $_POST['probation_period'] ) ) {
            $probation_period = absint( $_POST['probation_period'] );
            $probation_period = min( 24, max( 0, $probation_period ) );
            update_post_meta( $employee_id, 'hrm_probation_period', $probation_period );
        }
        
        if ( isset( $_POST['leave_applicability'] ) ) {
            $leave_applicability = sanitize_key( $_POST['leave_applicability'] );
            if ( ! in_array( $leave_applicability, [ 'during_probation', 'after_probation' ] ) ) {
                $leave_applicability = 'after_probation';
            }
            update_post_meta( $employee_id, 'hrm_leave_applicability', $leave_applicability );
        }
        
        // Handle Profile Picture
        $current_profile_pic = isset( $_POST['current_profile_pic'] ) ? sanitize_text_field( $_POST['current_profile_pic'] ) : '';
        
        // Check if user wants to remove the current profile picture
        if ( $current_profile_pic === 'remove' ) {
            delete_post_meta( $employee_id, 'hrm_profile_pic' );
        }
        
        // Handle new Profile Picture Upload
        if ( ! empty( $_FILES['profile_picture']['name'] ) ) {
            $attachment_id = self::handle_profile_picture_upload( $employee_id, $_FILES['profile_picture'] );
            if ( $attachment_id && ! is_wp_error( $attachment_id ) ) {
                update_post_meta( $employee_id, 'hrm_profile_pic', $attachment_id );
            }
        }

        wp_send_json_success( 'Employee updated successfully.' );
    }

    public static function handle_ajax_delete_employee() {
        if ( ! current_user_can( 'manage_options' ) ) wp_send_json_error( 'Unauthorized' );

        $employee_id = absint( $_POST['employee_id'] );
        $employee = get_post( $employee_id );
        if ( ! $employee ) wp_send_json_error( 'Not found' );

        $user_id = $employee->post_author;
        
        // Delete Data
        global $wpdb;
        $wpdb->delete( $wpdb->prefix . 'hrm_tasks', ['assigned_to' => $employee_id] );
        $wpdb->delete( $wpdb->prefix . 'hrm_daily_progress', ['employee_id' => $employee_id] );
        $wpdb->delete( $wpdb->prefix . 'hrm_attendance', ['employee_id' => $employee_id] );

        wp_delete_post( $employee_id, true );
        
        // Only delete the WP User if they are an employee and not the current admin
        if ( $user_id && $user_id != get_current_user_id() ) {
            // Additional check: Does this user have the employee role?
            $user = get_userdata( $user_id );
            if ( $user && in_array( 'hrm_employee', (array) $user->roles ) ) {
                wp_delete_user( $user_id );
            }
        }

        wp_send_json_success( 'User deleted successfully.' );
    }

    public static function handle_ajax_sync_employees() {
        check_ajax_referer( 'hrm_ajax_nonce', 'nonce' );
        if ( ! current_user_can( 'manage_options' ) ) wp_send_json_error( 'Unauthorized' );
        
        // Find users with role 'hrm_employee'
        $users = get_users(['role' => 'hrm_employee']);
        $count = 0;
        
        foreach($users as $user) {
            // Check if connected post exists
            $connected = get_posts([
                'post_type' => HRM_Setup::EMPLOYEE_POST_TYPE,
                'author' => $user->ID,
                'post_status' => 'any'
            ]);
            
            if(empty($connected)) {
                // Create post
                wp_insert_post([
                    'post_title' => $user->display_name ?: $user->user_login,
                    'post_type' => HRM_Setup::EMPLOYEE_POST_TYPE,
                    'post_status' => 'publish',
                    'post_author' => $user->ID
                ]);
                
                // Try to recover meta if possible, or just init
                update_post_meta( $connected, 'hrm_email', $user->user_email );
                
                $count++;
            }
        }
        
        if ($count > 0) {
            wp_send_json_success("Synced $count orphan users.");
        } else {
            wp_send_json_success("No orphan users found.");
        }
    }

    public static function handle_ajax_toggle_status() {
        if ( ! current_user_can( 'manage_options' ) ) wp_send_json_error( 'Unauthorized' );
        $id = absint( $_POST['employee_id'] );
        $status = sanitize_key( $_POST['status'] );
        wp_update_post([ 'ID' => $id, 'post_status' => $status ]);
        wp_send_json_success();
    }

    public static function handle_ajax_reset_password() {
         if ( ! current_user_can( 'manage_options' ) ) wp_send_json_error( 'Unauthorized' );
         $id = absint( $_POST['employee_id'] );
         $pass = $_POST['password'];
         $post = get_post($id);
         if($post && $post->post_author) {
             wp_set_password($pass, $post->post_author);
             wp_send_json_success('Password reset.');
         }
         wp_send_json_error('User not found.');
    }

    // --- Frontend Handlers ---

    public static function handle_frontend_profile_update( $employee_id ) {
        if ( ! isset( $_POST['hrm_action'] ) || $_POST['hrm_action'] !== 'update_profile' ) return;
        check_admin_referer( 'hrm_profile_action', 'hrm_profile_nonce' );

        $user_id = get_current_user_id();
        
        $phone = sanitize_text_field( $_POST['phone'] );
        $address = sanitize_textarea_field( $_POST['address'] );
        $email = sanitize_email( $_POST['email'] );

        // 1. Update Basic Meta
        update_post_meta( $employee_id, 'hrm_phone', $phone );
        update_post_meta( $employee_id, 'hrm_address', $address );
        update_post_meta( $employee_id, 'hrm_email', $email );

        // 2. Update WP User Email
        if ( is_email( $email ) && ! email_exists( $email ) ) {
                wp_update_user([ 'ID' => $user_id, 'user_email' => $email ]);
        } elseif ( $email !== wp_get_current_user()->user_email ) {
            // Email exists or invalid
        }

        // 3. Handle Password Change
        if ( ! empty( $_POST['current_password'] ) && ! empty( $_POST['new_password'] ) ) {
            $current_pass = $_POST['current_password'];
            $new_pass = $_POST['new_password'];
            $confirm_pass = $_POST['confirm_password'];
            $user = get_user_by( 'id', $user_id );

            if ( $user && wp_check_password( $current_pass, $user->user_pass, $user->ID ) ) {
                if ( $new_pass === $confirm_pass ) {
                    wp_set_password( $new_pass, $user_id );
                    // Re-login after password change
                    $creds = [
                        'user_login'    => $user->user_login,
                        'user_password' => $new_pass,
                        'remember'      => true
                    ];
                    wp_signon( $creds, false );
                } else {
                    wp_redirect( add_query_arg( 'profile_error', 'password_mismatch', wp_get_referer() ) );
                    exit;
                }
            } else {
                wp_redirect( add_query_arg( 'profile_error', 'invalid_password', wp_get_referer() ) );
                exit;
            }
        }

        // 4. Handle Profile Pic
        if ( ! empty( $_FILES['profile_pic']['name'] ) ) {
            require_once( ABSPATH . 'wp-admin/includes/image.php' );
            require_once( ABSPATH . 'wp-admin/includes/file.php' );
            require_once( ABSPATH . 'wp-admin/includes/media.php' );

            // Temporarily grant upload permission
            $user = wp_get_current_user();
            $added_cap = false;
            if ( ! $user->has_cap('upload_files') ) {
                $user->add_cap('upload_files');
                $added_cap = true;
            }

            $attachment_id = media_handle_upload( 'profile_pic', $employee_id );

            if ( $added_cap ) {
                $user->remove_cap('upload_files');
            }

            if ( ! is_wp_error( $attachment_id ) ) {
                update_post_meta( $employee_id, 'hrm_profile_pic', $attachment_id );
            }
        }

        wp_redirect( add_query_arg( 'profile_updated', '1', wp_get_referer() ) );
        exit;
    }

    public static function render_frontend_profile( $employee_id ) {
        $phone = get_post_meta( $employee_id, 'hrm_phone', true );
        $address = get_post_meta( $employee_id, 'hrm_address', true );
        $email = get_post_meta( $employee_id, 'hrm_email', true );
        $profile_pic_id = get_post_meta( $employee_id, 'hrm_profile_pic', true );
        $user_id = get_post_field( 'post_author', $employee_id );
        
        // Sync email if missing in meta
        if ( empty($email) && $user_id ) {
            $user = get_user_by('id', $user_id);
            $email = $user ? $user->user_email : '';
        }

        ?>
        <div class="hrm-profile-box" style="border: none; box-shadow: none; padding: 0;">
            <h3 style="font-size: 20px; font-weight: 700; margin-bottom: 32px; color: #111827; border: none; padding: 0;">My Profile</h3>

            <?php if ( isset( $_GET['profile_updated'] ) ) : ?>
                <div class="hrm-content-notice success">Profile updated successfully!</div>
            <?php endif; ?>
            
            <?php if ( isset( $_GET['profile_error'] ) ) : ?>
                <div class="hrm-content-notice error">
                    <?php 
                    if ($_GET['profile_error'] === 'password_mismatch') echo 'New passwords do not match.';
                    if ($_GET['profile_error'] === 'invalid_password') echo 'Current password is incorrect.';
                    ?>
                </div>
            <?php endif; ?>

            <form method="post" class="hrm-form-grid" enctype="multipart/form-data">
                <?php wp_nonce_field( 'hrm_profile_action', 'hrm_profile_nonce' ); ?>
                <input type="hidden" name="hrm_action" value="update_profile">
                
                <!-- Personal Information -->
                <h4 style="font-size: 14px; font-weight: 600; color: #4B5563; margin-bottom: 16px;">Personal Information</h4>
                
                <div class="hrm-form-group full-width" style="text-align: center; margin-bottom: 32px;">
                    <div class="hrm-profile-upload" style="position: relative; width: 120px; height: 120px; margin: 0 auto;">
                        <label for="profile_pic_input" style="cursor: pointer; display: block;">
                            <?php if ( $profile_pic_id ) : 
                                echo wp_get_attachment_image( $profile_pic_id, 'thumbnail', false, ['class' => 'profile-avatar-preview', 'style' => 'width: 120px; height: 120px; border-radius: 50%; object-fit: cover; border: 4px solid white; box-shadow: 0 4px 6px -1px rgba(0,0,0,0.1);'] );
                            else : ?>
                                <div class="profile-avatar-placeholder" style="width: 120px; height: 120px; border-radius: 50%; background: #E0E7FF; color: #3730A3; display: flex; align-items: center; justify-content: center; font-size: 48px; font-weight: 700; border: 4px solid white; box-shadow: 0 4px 6px -1px rgba(0,0,0,0.1);">
                                    <?php echo strtoupper(substr(get_the_title($employee_id), 0, 1)); ?>
                                </div>
                            <?php endif; ?>
                            <div class="upload-btn" style="position: absolute; bottom: 0; left: 50%; transform: translateX(-50%) translateY(50%); background: #059669; color: white; padding: 6px 16px; border-radius: 20px; font-size: 12px; font-weight: 600; display: flex; align-items: center; gap: 6px; box-shadow: 0 2px 4px rgba(0,0,0,0.1); white-space: nowrap;">
                                <span class="dashicons dashicons-camera" style="font-size: 16px; width: 16px; height: 16px;"></span> Change
                            </div>
                        </label>
                        <input type="file" name="profile_pic" id="profile_pic_input" accept="image/*" style="display: none;" onchange="previewImage(this)">
                    </div>
                </div>

                <div class="hrm-form-group full-width" style="margin-bottom: 24px;">
                    <label style="display: block; font-weight: 600; margin-bottom: 8px; font-size: 14px; color: #374151;">Full Name</label>
                    <input type="text" value="<?php echo esc_attr( get_the_title($employee_id) ); ?>" readonly disabled style="width: 100%; background: #F9FAFB; border: 1px solid #E5E7EB; border-radius: 8px; padding: 12px; font-size: 14px; color: #111827;">
                </div>

                <!-- Contact Details -->
                <h4 style="font-size: 14px; font-weight: 600; color: #4B5563; margin: 32px 0 16px;">Contact Details</h4>

                <div class="hrm-form-group full-width" style="margin-bottom: 20px;">
                    <label style="display: block; font-weight: 600; margin-bottom: 8px; font-size: 14px; color: #374151;">Email Address</label>
                    <input type="email" name="email" value="<?php echo esc_attr( $email ); ?>" required style="width: 100%; background: #F9FAFB; border: 1px solid #E5E7EB; border-radius: 8px; padding: 12px; font-size: 14px; color: #111827;">
                </div>

                <div class="hrm-form-group full-width" style="margin-bottom: 20px;">
                    <label style="display: block; font-weight: 600; margin-bottom: 8px; font-size: 14px; color: #374151;">Phone Number</label>
                    <input type="text" name="phone" value="<?php echo esc_attr( $phone ); ?>" placeholder="+1 234 567 890" style="width: 100%; background: #F9FAFB; border: 1px solid #E5E7EB; border-radius: 8px; padding: 12px; font-size: 14px; color: #111827;">
                </div>

                <div class="hrm-form-group full-width" style="margin-bottom: 24px;">
                    <label style="display: block; font-weight: 600; margin-bottom: 8px; font-size: 14px; color: #374151;">Address</label>
                    <textarea name="address" rows="3" placeholder="Enter your address" style="width: 100%; background: #F9FAFB; border: 1px solid #E5E7EB; border-radius: 8px; padding: 12px; font-size: 14px; color: #111827; resize: vertical;"><?php echo esc_textarea( $address ); ?></textarea>
                </div>

                <!-- Security Settings -->
                <h4 style="font-size: 14px; font-weight: 600; color: #4B5563; margin: 32px 0 16px;">Security Settings</h4>

                <div class="hrm-form-group full-width" style="margin-bottom: 20px;">
                    <label style="display: block; font-weight: 600; margin-bottom: 8px; font-size: 14px; color: #374151;">Current Password</label>
                    <input type="password" name="current_password" placeholder="********" style="width: 100%; background: #F9FAFB; border: 1px solid #E5E7EB; border-radius: 8px; padding: 12px; font-size: 14px; color: #111827;">
                </div>
                
                <div class="hrm-form-group full-width" style="margin-bottom: 20px;">
                    <label style="display: block; font-weight: 600; margin-bottom: 8px; font-size: 14px; color: #374151;">New Password</label>
                    <input type="password" name="new_password" placeholder="********" style="width: 100%; background: #F9FAFB; border: 1px solid #E5E7EB; border-radius: 8px; padding: 12px; font-size: 14px; color: #111827;">
                </div>

                <div class="hrm-form-group full-width" style="margin-bottom: 32px;">
                    <label style="display: block; font-weight: 600; margin-bottom: 8px; font-size: 14px; color: #374151;">Confirm New Password</label>
                    <input type="password" name="confirm_password" placeholder="********" style="width: 100%; background: #F9FAFB; border: 1px solid #E5E7EB; border-radius: 8px; padding: 12px; font-size: 14px; color: #111827;">
                </div>

                <div class="hrm-form-group full-width">
                    <button type="submit" class="hrm-btn-primary" style="width: 100%; justify-content: center; padding: 14px; font-size: 16px; border-radius: 8px;">
                        <span class="dashicons dashicons-saved"></span> Save Changes
                    </button>
                </div>
            </form>

            <script>
            function previewImage(input) {
                if (input.files && input.files[0]) {
                    var reader = new FileReader();
                    reader.onload = function(e) {
                        var img = document.querySelector('.profile-avatar-preview');
                        var placeholder = document.querySelector('.profile-avatar-placeholder');
                        
                        if (img) {
                            img.src = e.target.result;
                        } else if (placeholder) {
                            // Replace placeholder with image
                            var newImg = document.createElement('img');
                            newImg.src = e.target.result;
                            newImg.className = 'profile-avatar-preview';
                            newImg.style = 'width: 120px; height: 120px; border-radius: 50%; object-fit: cover; border: 4px solid white; box-shadow: 0 4px 6px -1px rgba(0,0,0,0.1);';
                            placeholder.parentNode.replaceChild(newImg, placeholder);
                        }
                    }
                    reader.readAsDataURL(input.files[0]);
                }
            }
            </script>
        </div>
        <?php
    }
}
